C ABCDEFGHIJKLMNOPQRSTUVWXYZ$0123456789+-*/=(),.
C
C                        MP  (VERSION 810614)
C                        ********************
C
C                 ***********************************
C                 * COPYRIGHT (C) 1980, R. P. BRENT *
C                 ***********************************
C
C $$                   ******  COMMENTS  ******
C
C MP IS A MULTIPLE-PRECISION FLOATING-POINT ARITHMETIC PACKAGE.
C IT IS ALMOST COMPLETELY MACHINE-INDEPENDENT, AND SHOULD
C RUN ON ANY MACHINE WITH AN ANSI STANDARD FORTRAN COMPILER
C AND SUFFICIENT MEMORY.
C
C FOR A GENERAL DESCRIPTION OF THE PHILOSOPHY AND DESIGN OF MP,
C SEE - R. P. BRENT, A FORTRAN MULTIPLE-PRECISION ARITHMETIC
C PACKAGE, ACM TRANS. MATH. SOFTWARE 4 (MARCH 1978), 57-70.
C SOME ADDITIONAL DETAILS ARE GIVEN IN THE SAME ISSUE, 71-81.
C FOR DETAILS OF THE IMPLEMENTATION, CALLING SEQUENCES ETC. SEE
C THE MP USERS GUIDE DISTRIBUTED WITH THE PACKAGE.
C
C MP IS NORMALLY DISTRIBUTED IN SIX FILES.  ALL HAVE 80 CHARACTER
C LOGICAL RECORDS AND USE ONLY THE (STANDARD FORTRAN) CHARACTERS
C APPEARING IN THE FIRST LINE ABOVE, EXCEPT FOR INEQUALITY SIGNS AND
C THE QUOTE CHARACTER IN FILE 6, AND THESE PLUS THE UNDERSCORE
C CHARACTER AND LOWER CASE LETTERS IN FILE 4 (THE MP USERS GUIDE).
C THE TAPE IS NORMALLY UNLABELLED, 9 TRACK, EBCDIC AND 8-BIT ASCII,
C FIXED LENGTH (80 CHARACTER) LOGICAL RECORDS (NO CR LF),
C BLOCKING FACTOR 12, EITHER 800 OR 1600 FPI, ODD PARITY.
C THE FILES ON THE TAPE ARE AS FOLLOWS -
C
C FILE 1 - THESE COMMENTS AND EXAMPLE PROGRAM.
C FILE 2 - MP SUBROUTINES (EXCLUDING EXAMPLE AND TEST PROGRAMS).
C FILE 3 - TEST PROGRAMS (NOT USING AUGMENT INTERFACE).
C FILE 4 - MP USERS GUIDE (DUO-CASE VERSION).
C FILE 5 - AUGMENT DESCRIPTION DECK AND JACOBI PROGRAM USING IT.
C          (MP MAY BE USED WITH THE AUGMENT PREPROCESSOR.  FOR
C           DETAILS SEE SECTION 4 OF THE USERS GUIDE.)
C FILE 6 - MP USERS GUIDE (UPPER-CASE VERSION).
C FILES 7-12 - SAME AS FILES 1-6 EXCEPT 1-6 ARE EBCDIC, 7-12 ASCII.
C
C TO INSTALL MP, READ THESE 6 FILES.  PRINT FILE 4 OR 6 (USERS GUIDE)
C USING THE FIRST CHARACTER (BLANK, +, 0 OR 1) AS STANDARD FORTRAN
C PRINTER CONTROL.  THEN FOLLOW THE INSTRUCTIONS GIVEN IN SECTION
C 9.1 OF THE USERS GUIDE.
C
C CORRESPONDENCE CONCERNING MP SHOULD BE SENT TO -
C
C                PROF. R. P. BRENT,
C                DEPARTMENT OF COMPUTER SCIENCE,
C                AUSTRALIAN NATIONAL UNIVERSITY,
C                BOX 4, CANBERRA, ACT 2600,
C                AUSTRALIA.
C
C $$                   ******  EXAMPLE  ******
C
C THIS PROGRAM COMPUTES PI AND EXP(PI*SQRT(163/9)) TO 100
C DECIMAL PLACES, AND EXP(PI*SQRT(163)) TO 90 DECIMAL PLACES,
C AND WRITES THEM ON LOGICAL UNIT 6.
C
C CORRECT OUTPUT (EXCLUDING HEADINGS) IS AS FOLLOWS
C
C                  3.14159265358979323846264338327950288419716939937510
C                    58209749445923078164062862089986280348253421170680
C             640320.00000000060486373504901603947174181881853947577148
C                    57603665918194652218258286942536340815822646477590
C 262537412640768743.99999999999925007259719818568887935385633733699086
C                    2707537410378210647910118607312951181346
C
C WORKING SPACE IN BLANK COMMON, PARAMETERS IN
C COMMON /MPCOM/ (WHICH HAS LENGTH 23)
      COMMON R
      COMMON /MPCOM/ B, T, M, LUN, MXR, SPTR, MXSPTR, DUMMY
      INTEGER B, T, M, LUN, MXR, SPTR, MXSPTR, DUMMY(16), R(500)
C
C WE HAVE T .LE. 62 IF WORDLENGTH AT LEAST 16 BITS, AND WORKING SPACE
C IS AT MOST 500 WORDS (LESS IF WORDLENGTH IS GREATER THAN 16 BITS).
C
C VARIABLES NEED T+2 .LE. 64 WORDS AND ALLOW 110 CHARACTERS FOR
C DECIMAL OUTPUT
      INTEGER PI(64), X(64), C(110)
C
C CALL MPSET2 TO SET OUTPUT LOGICAL UNIT = 6 AND EQUIVALENT
C NUMBER OF DECIMAL PLACES TO AT LEAST 110.  THE LAST THREE
C PARAMETERS ARE THE DIMENSIONS OF PI (OR X) AND THE LOWER
C AND UPPER INDICES OF BLANK COMMON AVAILABLE TO MP.
      CALL MPSET2 (6, 110, 64, 1, 500)
C
C COMPUTE MULTIPLE-PRECISION PI
      CALL MPPI(PI)
C
C CONVERT TO PRINTABLE FORMAT (F110.100) AND WRITE
      CALL MPOUT (PI, C, 110, 100)
      WRITE (LUN, 10) B, T, C
   10 FORMAT (32H1EXAMPLE OF MP PACKAGE,   BASE =, I9,
     $  12H,   DIGITS =, I4 /// 11H PI TO 100D //
     $  11X, 60A1 / 21X, 50A1)
C
C SET X = SQRT(163/9), THEN MULTIPLY BY PI
      CALL MPQPWR (163, 9, 1, 2, X)
      CALL MPMUL (X, PI, X)
C
C SET X = EXP(X)
      CALL MPEXP (X, X)
C
C CONVERT TO PRINTABLE FORMAT AND WRITE
      CALL MPOUT (X, C, 110, 100)
      WRITE (LUN, 20) C
   20 FORMAT (/ 28H EXP(PI*SQRT(163/9)) TO 100D //
     $        11X, 60A1 / 21X, 50A1)
C
C SET X = X**3 = EXP(PI*SQRT(163))
      CALL MPPWR (X, 3, X)
C
C WRITE IN FORMAT F110.90
      CALL MPOUT (X, C, 110, 90)
      WRITE (LUN, 30) C
   30 FORMAT (/ 25H EXP(PI*SQRT(163)) TO 90D //
     $        1X, 70A1 / 21X, 40A1)
      WRITE (LUN, 40) MXSPTR
   40 FORMAT (/ 21H END OF EXAMPLE, USED, I4,
     $          23H WORDS OF WORKING SPACE //)
      STOP
      END
C $$                   ******  MPABS  ******
      SUBROUTINE MPABS (X, Y)
C SETS Y = ABS(X) FOR MP NUMBERS X AND Y.
C Y WILL BE PACKED IF X IS.
      INTEGER X(1), Y(1)
      CALL MPSTR (X, Y)
      Y(1) = IABS(Y(1))
      RETURN
      END
C $$                   ******  MPADD  ******
      SUBROUTINE MPADD (X, Y, Z)
C ADDS X AND Y, FORMING RESULT IN Z, WHERE X, Y AND Z ARE MP NUMBERS.
C ROUNDING DEFINED BY PARAMETER RNDRL IN COMMON /MPCOM/ AS FOLLOWS -
C RNDRL = 0 - TRUNCATE TOWARDS ZERO IF X*Y .GE. 0,
C             AWAY FROM ZERO IF X*Y .LT. 0,
C             IN BOTH CASES USING ONE GUARD DIGIT, SO RESULT IS
C             EXACT IF SEVERE CANCELLATION OCCURS.
C RNDRL = 1, 2 OR 3 -  SEE COMMENTS IN SUBROUTINE MPNZR.
C             SUFFICIENT GUARD DIGITS ARE USED TO ENSURE THE
C             CORRECT RESULT.
      COMMON R
      COMMON /MPCOM/ B, T, M, LUN, MXR, SPTR, MXSPTR, MNSPTR,
     $               MXEXPN, MNEXPN, RNDRL, DUMMY
      INTEGER ED, I2, J, KG, MED, RE, RS, S, TWO
      INTEGER B, DUMMY(12), LUN, M, MNEXPN, MNSPTR, MXEXPN, MXR, MXSPTR,
     $     R(1), RNDRL, SPTR, T, X(1), Y(1), Z(1)
      TWO = 2
C COMPUTE NUMBER OF GUARD DIGITS REQUIRED
      KG = 1
      IF (RNDRL.NE.0) KG = T
C ALLOCATE T + KG WORDS FOR ACCUMULATOR.
      CALL MPNEW2 (I2, T+KG)
C CHECK FOR X OR Y ZERO
      IF (X(1).NE.0) GO TO 20
C X = 0 OR NEGLIGIBLE, SO RESULT = Y
   10 CALL MPSTR(Y, Z)
      GO TO 120
   20 IF (Y(1).NE.0) GO TO 40
C Y = 0 OR NEGLIGIBLE, SO RESULT = X
   30 CALL MPSTR (X, Z)
      GO TO 120
C COMPARE SIGNS
   40 S = X(1)*Y(1)
      IF (IABS(S).LE.1) GO TO 50
      CALL MPERRM (38HSIGN NOT 0, +1 OR -1 IN CALL TO MPADD$)
      GO TO 80
C COMPARE EXPONENTS
   50 ED = X(TWO) - Y(TWO)
      MED = IABS(ED)
C CAN REDUCE KG IF MED SMALL.  THIS SAVES TIME.
      KG = MIN0 (KG, MED + (S+1)/2)
      IF (ED) 90, 60, 130
C EXPONENTS EQUAL SO COMPARE SIGNS, THEN FRACTIONS IF NEC.
   60 IF (S.GT.0) GO TO 100
      DO 70 J = 1, T
      IF (X(J+2) - Y(J+2)) 100, 70, 140
   70 CONTINUE
C RESULT IS ZERO
   80 Z(1) = 0
      GO TO 120
C HERE EXPONENT(Y) .GT. EXPONENT(X)
   90 IF ((MED.GT.T).AND.(RNDRL.LE.1)) GO TO 10
C CHECK FOR COMMON SPECIAL CASE HERE (FOR SAKE OF SPEED)
      IF ((RNDRL.GT.0).OR.(((Y(TWO+1).GE.(B-1)).OR.(S.LT.0)).AND.
     $     ((Y(TWO+1).EQ.1).OR.(S.GT.0))))  GO TO 100
C CAN AVOID CALLING MPNZR HERE, SAVING TIME.
      CALL MPADD3 (X, Y, RE, Z(TWO+1), 0)
      Z(1) = Y(1)
      Z(TWO) = RE
      GO TO 120
C HERE ABS(Y) .GT. ABS(X)
  100 RS = Y(1)
      CALL MPADD3 (X, Y, RE, R(I2), KG)
C NORMALIZE AND ROUND OR TRUNCATE
  110 CALL MPNZR (RS, RE, Z, R(I2), KG)
C RESTORE STACK POINTER AND RETURN
  120 SPTR = I2
      RETURN
C HERE EXPONENT(X) .GT. EXPONENT(Y)
C CODE IS AS ABOVE WITH X AND Y INTERCHANGED.
  130 IF ((MED.GT.T).AND.(RNDRL.LE.1)) GO TO 30
C CHECK FOR COMMON SPECIAL CASE HERE (FOR SAKE OF SPEED)
      IF ((RNDRL.GT.0).OR.(((X(TWO+1).GE.(B-1)).OR.(S.LT.0)).AND.
     $     ((X(TWO+1).EQ.1).OR.(S.GT.0))))  GO TO 140
C CAN AVOID CALLING MPNZR HERE.
      CALL MPADD3 (Y, X, RE, Z(TWO+1), 0)
      Z(1) = X(1)
      Z(TWO) = RE
      GO TO 120
C HERE ABS(X) .GT. ABS(Y)
  140 RS = X(1)
      CALL MPADD3 (Y, X, RE, R(I2), KG)
      GO TO 110
      END
C $$                   ******  MPADDI  ******
      SUBROUTINE MPADDI (X, IY, Z)
C ADDS MULTIPLE-PRECISION X TO INTEGER IY GIVING MULTIPLE-PRECISION Z.
C ROUNDING IS CONTROLLED BY RNDRL IN COMMON /MPCOM/ - SEE
C COMMENTS IN SUBROUTINE MPNZR.
      COMMON R
      INTEGER IY, I2, R(1), X(1), Z(1), SV
C SAVE T ETC., ALLOCATE TEMPORARY SPACE.
      CALL MPSAVN (SV)
      CALL MPNEW (I2)
C CONVERT IY TO MULTIPLE-PRECISION AND ADD TO X.
      CALL MPCIM (IY, R(I2))
      CALL MPADD (X, R(I2), Z)
C RESTORE EVERYTHING.
      CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPADDQ  ******
      SUBROUTINE MPADDQ (X, I, J, Y)
C ADDS THE RATIONAL NUMBER I/J TO MP NUMBER X, MP RESULT IN Y
C ROUNDING DEFINED BY PARAMETER RNDRL IN COMMON /MPCOM/ -
C EFFECT IS SAME AS CONVERTING I/J TO MULTIPLE-PRECISION USING
C MPCQM AND THEN ADDING TO X USING MPADD, SO SEE COMMENTS IN
C THESE ROUTINES.
      COMMON R
      INTEGER I, I2, J, R(1), SV, X(1), Y(1)
      CALL MPSAVN (SV)
C ALLOCATE TEMPORARY SPACE, COMPUTE I/J AND ADD TO X
      CALL MPNEW (I2)
      CALL MPCQM (I, J, R(I2))
      CALL MPADD (X, R(I2), Y)
      CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPADD3  ******
      SUBROUTINE MPADD3 (X, Y, RE, A, KG)
C CALLED BY MPADD, DOES INNER LOOPS OF ADDITION
C ASSUMES 0 .LT. ABS(X) .LE. ABS(Y).
C RETURNS DIGITS OF X + Y IN A(1), ... , A(T+KG)
C AND EXPONENT IN RE.  ASSUMES KG .GE. 0.  THERE IS A SLIGHT
C CLUDGE TO ENSURE CORRECT DIRECTED ROUNDING.  THIS MAY
C AFFECT A(T+1), ... , A(T+KG).
      COMMON /MPCOM/ B, T, DUMMY
      INTEGER C, I, J, MED, S, TED, TG, TG1, TWO,
     $  A(1), B, DUMMY(21), KG, RE, T, X(1), Y(1)
      TWO = 2
C Y(TWO) - X(TWO) .GT. T IN CALL ONLY FOR RNDRL .GT. 1
      MED = MIN0 (Y(TWO) - X(TWO), T)
      TED = T + MED
      S = X(1)*Y(1)
      RE = Y(TWO)
      TG = T + KG
      I = TG
      C = 0
C CLEAR GUARD DIGITS TO RIGHT OF X DIGITS
   10 IF (I.LE.TED) GO TO 20
      A(I) = 0
      I = I - 1
      GO TO 10
   20 IF (S.LT.0) GO TO 130
C HERE DO ADDITION, EXPONENT(Y) .GE. EXPONENT(X)
      IF (I.LE.T) GO TO 40
   30 J = I - MED
      A(I) = X(J+2)
      I = I - 1
      IF (I.GT.T) GO TO 30
   40 IF (I.LE.MED) GO TO 60
      J = I - MED
      C = Y(I+2) + X(J+2) + C
      IF (C.LT.B) GO TO 50
C CARRY GENERATED HERE
      A(I) = C - B
      C = 1
      I = I - 1
      GO TO 40
C NO CARRY GENERATED HERE
   50 A(I) = C
      C = 0
      I = I - 1
      GO TO 40
   60 IF (I.LE.0) GO TO 90
      C = Y(I+2) + C
      IF (C.LT.B) GO TO 70
      A(I) = 0
      C = 1
      I = I - 1
      GO TO 60
   70 A(I) = C
      I = I - 1
C NO CARRY POSSIBLE HERE
   80 IF (I.LE.0) RETURN
      A(I) = Y(I+2)
      I = I - 1
      GO TO 80
   90 IF (C.EQ.0) RETURN
C MUST SHIFT RIGHT HERE AS CARRY OFF END
      TG1 = TG + 1
      DO 100 J = 2, TG
      I = TG1 - J
  100 A(I+1) = A(I)
      A(1) = 1
      RE = RE + 1
      RETURN
C HERE DO SUBTRACTION, ABS(Y) .GT. ABS(X)
  110 J = I - MED
      A(I) = C - X(J+2)
      C = 0
      IF (A(I).GE.0) GO TO 120
C BORROW GENERATED HERE
      C = -1
      A(I) = A(I) + B
  120 I = I - 1
  130 IF (I.GT.T) GO TO 110
  140 IF (I.LE.MED) GO TO 160
      J = I - MED
      C = Y(I+2) + C - X(J+2)
      IF (C.GE.0) GO TO 150
C BORROW GENERATED HERE
      A(I) = C + B
      C = -1
      I = I - 1
      GO TO 140
C NO BORROW GENERATED HERE
  150 A(I) = C
      C = 0
      I = I - 1
      GO TO 140
  160 IF (I.LE.0) RETURN
      C = Y(I+2) + C
      IF (C.GE.0) GO TO 70
      A(I) = C + B
      C = -1
      I = I - 1
      GO TO 160
      END
C $$                   ******  MPART1  ******
      SUBROUTINE MPART1 (N, Y)
C COMPUTES MP Y = ARCTAN(1/N), ASSUMING INTEGER N .GT. 1.
C USES SERIES ARCTAN(X) = X - X**3/3 + X**5/5 - ...
C ROUNDING IS DEFINED BY THE PARAMETER RNDRL IN COMMON /MPCOM/ -
C RNDRL = 0 OR 1 - ERROR LESS THAN 0.6 UNITS IN LAST PLACE.
C RNDRL = 2 OR 3 - SEE COMMENTS IN SUBROUTINE MPNZR.
      COMMON R
      COMMON /MPCOM/ B, T, M, LUN, MXR, SPTR, MXSPTR, MNSPTR,
     $  MXEXPN, MNEXPN, RNDRL, KTUNFL, DUMMY
      INTEGER I, ID, I2, I3, KTU, SV, TG
      INTEGER B, DUMMY(11), KTUNFL, LUN, M, MNEXPN, MNSPTR, MXEXPN, MXR
      INTEGER MXSPTR, N, R(1), RNDRL, SPTR, T, Y(1)
      INTEGER MPPARN, MPTLB
C SAVE T ETC.
      CALL MPSAVN (SV)
      IF (N .LE. 1) CALL MPERRM (27HN .LE. 1 IN CALL TO MPART1$)
C INCREASE WORKING PRECISION.
      IF ((N.LT.B).AND.(T.GT.2)) T = T - 1
      CALL MPGD3 (MPTLB (IABS(R(SV))), TG)
C SAVE UNDERFLOW COUNTER
      KTU = KTUNFL
C USE TRUNCATED ARITHMETIC INTERNALLY IF RNDRL .LE. 1.
      IF (RNDRL.EQ.1) RNDRL = 0
C ALLOCATE SPACE FOR TEMPORARY STORAGE
      CALL MPNEW (I2)
      CALL MPNEW (I3)
C SET SUM TO 1/N
      CALL MPCQM (1, N, R(I3))
C SET ADDITIVE TERM TO 1/N
      CALL MPSTR (R(I3), R(I2))
      I = 1
C MPPARN(16) RETURNS A LARGE INTEGER (SEE MPSET).
      ID = (MPPARN(16)/N)/N - 2
C MAIN LOOP.  FIRST REDUCE T IF POSSIBLE
   10 T = TG + 2 + R(I2+1) - R(I3+1)
      IF (T.LT.2) GO TO 40
      T = MIN0 (T, TG)
      IF (RNDRL.NE.0) T = TG
C IF (I+2)*N**2 IS NOT REPRESENTABLE AS AN INTEGER THE DIVISION
C FOLLOWING HAS TO BE PERFORMED IN SEVERAL STEPS.
      IF (I.GE.ID)  GO TO 20
      CALL MPMULQ (R(I2), -I, (I+2)*N*N, R(I2))
      GO TO 30
   20 CALL MPMULQ (R(I2), -I, I+2, R(I2))
      CALL MPMULS (R(I2), 1, 1, N, N)
   30 I = I + 2
C RESTORE T TO WORKING PRECISION
      T = TG
C ACCUMULATE SUM
      CALL MPADD (R(I3), R(I2), R(I3))
C CAN ONLY FALL THROUGH NEXT STATEMENT IF MP UNDERFLOW OCCURED.
      IF (KTU.EQ.KTUNFL) GO TO 10
C RESTORE T ETC., ROUND RESULT AND RETURN
   40 CALL MPRES2 (SV)
      CALL MPRND (R(I3), TG, Y, IABS(T), 1)
      CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPCHGB  ******
      INTEGER FUNCTION MPCHGB (B1, B2, N)
C RETURNS J SUCH THAT B1**ABS(J) .GE. B2**ABS(N),
C I.E.    ABS(J) .GE. ABS(N)*LOG(B2)/LOG(B1),
C AND     SIGN(J) = SIGN(N),
C ASSUMING B1 .GT. 1, B2 .GE. 1, B1*B2 .LE. MXINT .
C USUALLY THE VALUE OF J RETURNED IS CLOSE TO MINIMAL.
      INTEGER B1, B2, J, K, MB, MPPARN, MUL, MXINT, N, N2, Q
      MXINT = MPPARN(16)
      MB = MXINT/B1
      IF ((B1 .LE. 1) .OR. (B2 .LE. 0) .OR. (B2 .GT. MB))
     $  CALL MPERRM (36HILLEGAL ARGUMENTS IN CALL TO MPCHGB$)
      J = 0
      IF ((N .EQ. 0) .OR. (B2 .EQ. 1)) GO TO 60
      K = 0
      Q = 1
C THE CONSTANT 100 IS LARGE ENOUGH TO GIVE REASONABLE ACCURACY.
C IF IT IS INCREASED THE TIME REQUIRED WILL INCREASE.
      MUL = (IABS(N)-1)/100 + 1
      N2 = (IABS(N)-1)/MUL + 1
C INCREASE J AND K, KEEPING Q .LE. B1**J/B2**K
   10 IF (K .GE. N2) GO TO 40
   20 IF (Q .GT. MB) GO TO 30
      Q = B1*Q
      J = J + 1
      GO TO 20
   30 Q = Q/B2
      K = K + 1
      GO TO 10
C FINAL STAGE, MAY HAVE OVERSHOT
   40 IF (Q .LT. B1) GO TO 50
      Q = Q/B1
      J = J - 1
      GO TO 40
C CHECK IF RESULT WOULD OVERFLOW
   50 IF (J .GT. (MXINT/MUL)) CALL MPERRM (
     $  30HN TOO LARGE IN CALL TO MPCHGB$)
      J = ISIGN (MUL*J, N)
   60 MPCHGB = J
      RETURN
      END
C $$                   ******  MPCHK  ******
      SUBROUTINE MPCHK
C CHECKS LEGALITY OF PARAMETERS WHICH SHOULD BE SET IN
C COMMON /MPCOM/, ALSO UPDATES MXSPTR.
      COMMON /MPCOM/ B, T, M, LUN, MXR, SPTR, MXSPTR, MNSPTR,
     $  MXEXPN, MNEXPN, RNDRL, KTUNFL, MXUNFL, DECPL, MT2,
     $  MXINT, EXWID, INRECL, INBASE, OUTBAS, EXPCH, CHWORD, ONESCP
      INTEGER B, CHWORD, DECPL, EXPCH, EXWID, INBASE, INRECL, KTUNFL,
     $  LUN, M, MNEXPN, MNSPTR, MT2, MXEXPN, MXINT, MXR, MXSPTR, MXUNFL,
     $  ONESCP, OUTBAS, RNDRL, SPTR, T, IB, JSP
C FIRST CHECK THAT LUN IN RANGE 1 TO 99, IF NOT PRINT ERROR
C MESSAGE ON LOGICAL UNIT 6.
      IF ((0.LT.LUN) .AND. (LUN.LT.100)) GO TO 20
      WRITE (6, 10) LUN
   10 FORMAT (10H *** LUN =, I10, 29H ILLEGAL IN CALL TO MPCHK ***)
      LUN = 6
      CALL MPERR
C NOW CHECK LEGALITY OF B, T AND M
   20 IF (B.GT.1) GO TO 40
      WRITE (LUN, 30) B
   30 FORMAT (8H *** B =, I10, 29H ILLEGAL IN CALL TO MPCHK ***)
      CALL MPERR
C T MAY EXCEED MT2-2 IN CALL FROM OTHER MP ROUTINES
   40 IF ((T.GT.1). AND. ((T.LE.(MT2-2)) .OR. (SPTR.GT.MNSPTR)))
     $   GO TO 60
      WRITE (LUN, 50) T
   50 FORMAT (8H *** T =, I10, 29H ILLEGAL IN CALL TO MPCHK ***)
      CALL MPERR
   60 IF (M.GT.(4*T)) GO TO 80
C HERE M IS NOT GREATER THAN 4*T.  SINCE MANY MP ROUTINES INCREASE
C T INTERNALLY, IT IS SAFEST TO SET M MUCH LARGER THAN 4*T
C INITIALLY.  NOTE, THOUGH, THAT 4*M MUST NOT OVERFLOW.
      WRITE (LUN, 70)
   70 FORMAT (36H *** M .LE. 4*T IN CALL TO MPCHK ***)
      CALL MPERR
C 8*B*B-1 SHOULD BE REPRESENTABLE, IF NOT WILL OVERFLOW
C AND MAY BECOME NEGATIVE, SO CHECK FOR THIS
   80 IB = 4*B*B - 1
      IF ((IB.GT.0) .AND. ((2*IB+1).GT.0)) GO TO 100
      WRITE (LUN, 90)
   90 FORMAT (37H *** B TOO LARGE IN CALL TO MPCHK ***)
      CALL MPERR
C CHECK THAT STACK SPACE IN COMMON IS SUFFICIENT.
C SPTR SHOULD POINT TO FIRST FREE WORD IN STACK,
C AND MXR TO LAST AVAILABLE WORD.
C AT LEAST 80 WORDS MUST BE AVAILABLE (SEE MPERRM).
  100 JSP = MAX0 (SPTR, MNSPTR + 80) - 1
      IF (MXR.GE.JSP) GO TO 120
C HERE STACK IS TOO SMALL.  CALL MPSTOV TO EXPAND IT IF POSSIBLE.
      CALL MPSTOV (JSP - MXR)
C SEE IF MPSTOV ACTUALLY DID EXPAND ENOUGH.
      IF (MXR.GE.JSP) GO TO 120
C HERE COMMON IS TOO SMALL (ASSUMING ITS SIZE IS MXR).
      WRITE (LUN, 110) MXR, JSP
  110 FORMAT (51H *** MXR TOO SMALL OR NOT SET TO DIM(R) BEFORE CALL,
     $ 21H TO AN MP ROUTINE *** /
     $ 10H *** MXR =, I6, 24H, BUT SHOULD BE AT LEAST, I6, 5H  ***)
      CALL MPERR
C CHECK LEGALITY OF VARIOUS OTHER PARAMETERS IN COMMON /MPCOM/
  120 IF ((MXSPTR.GE.MNSPTR) .AND. (MNSPTR.GT.0) .AND.
     $    (MNSPTR.LE.SPTR) .AND. ((MXSPTR-1).LE.MXR) .AND.
     $    (RNDRL.GE.0) .AND. (RNDRL.LE.3) .AND.
     $    (KTUNFL.GE.0) .AND. (MXUNFL.GE.0)) GO TO 140
      WRITE (LUN, 130)
  130 FORMAT (38H *** ONE OF SPTR, ... , MXUNFL ILLEGAL,
     $        21H IN CALL TO MPCHK ***)
      CALL MPERR
  140 IF ((DECPL .GT. 0) .AND. (MT2 .GT. 3) .AND. (MXINT .GE. 2047)
     $   .AND. (EXWID .GT. 2) .AND. (INRECL .GT. 0)
     $   .AND. (INRECL .LE. 80) .AND. (INBASE .GT. 1)
     $   .AND. (INBASE .LE. 16) .AND. (OUTBAS .GT. 1)
     $   .AND. (OUTBAS .LE. 16) .AND. (CHWORD .GT. 0)) GO TO 160
      WRITE (LUN, 150)
  150 FORMAT (39H *** ONE OF DECPL, ... , CHWORD ILLEGAL,
     $        21H IN CALL TO MPCHK ***)
      CALL MPERR
C UPDATE MXSPTR AND RETURN
  160 MXSPTR = MAX0 (MXSPTR, SPTR)
      RETURN
      END
C $$                   ******  MPCIM  ******
      SUBROUTINE MPCIM (IX, Z)
C CONVERTS INTEGER IX TO MULTIPLE-PRECISION Z.
C ASSUMES THAT ABS(IX) .LE. B**T (OTHERWISE IX CAN NOT USUALLY BE
C REPRESENTED EXACTLY AS A MULTIPLE-PRECISION NUMBER).
      COMMON /MPCOM/ B, T, DUMMY
      INTEGER B, DUMMY(21), I, IX, MPGD, N, T, TWO, Z(1)
      TWO = 2
C CHECK LEGALITY OF PARAMETERS IN COMMON /MPCOM/
      CALL MPCHK
C T SHOULD BE INCREASED IF ABS(IX) .LT. B**T .
      N = IX
      IF (MPGD(N).GT.T) CALL MPERRM (
     $  29HT TOO SMALL IN CALL TO MPCIM$)
C SET Z(1) TO SIGN OF IX.
      Z(1) = 0
      IF (N.EQ.0) RETURN
      Z(1) = ISIGN (1, N)
C SET EXPONENT TO T
      Z(TWO) = T
C CLEAR FRACTION
      DO 10 I = 2, T
   10 Z(I+1) = 0
C INSERT N
      Z(T+2) = IABS(N)
C NORMALIZE BY CALLING MPMULI
      CALL MPMULI (Z, 1, Z)
      RETURN
      END
C $$                   ******  MPCMF  ******
      SUBROUTINE MPCMF (X, Y)
C FOR MP X AND Y, RETURNS FRACTIONAL PART OF X IN Y,
C I.E., Y = X - INTEGER PART OF X (TRUNCATED TOWARDS 0).
C NOTE THAT RNDRL IN COMMON /MPCOM/ IS IRRELEVANT.
      COMMON /MPCOM/ B, T, DUMMY
      INTEGER B, DUMMY(21), I, T, TWO, X(1), Y(1), Y1, Y2
      TWO = 2
C MOVE X TO Y
      CALL MPSTR (X, Y)
C CHECK IF Y ZERO, WHEN RETURN WITH RESULT ZERO.
      Y1 = Y(1)
      IF (Y1.EQ.0) RETURN
C CAN ALSO RETURN IF EXPONENT NOT POSITIVE, THEN RESULT X.
      Y2 = Y(TWO)
      IF (Y2.LE.0) RETURN
C CHECK FOR ZERO FRACTIONAL PART
      IF (Y2.LT.T) GO TO 10
C HERE THE FRACTIONAL PART OF X IS ZERO.
      Y(1) = 0
      RETURN
C HERE 0 .LT. Y2 .LT. T.  CLEAR INTEGER PART.
   10 DO 20 I = 1, Y2
   20 Y(I+2) = 0
C NORMALIZE RESULT AND RETURN.
      CALL MPNZR (Y1, Y2, Y(1), Y(TWO+1), 0)
      RETURN
      END
C $$                   ******  MPCMI  ******
      SUBROUTINE MPCMI (X, IZ)
C CONVERTS MULTIPLE-PRECISION X TO INTEGER IZ,
C ASSUMING THAT X NOT TOO LARGE (ELSE USE MPCMIM).
C X IS TRUNCATED TOWARDS ZERO, REGARDLESS OF THE VALUE OF RNDRL.
C IZ IS RETURNED AS ZERO IF ABS(INT(X)) .GT. MXINT .
C THE USER MAY CHECK FOR THIS BY TESTING IF
C     ((X(1).NE.0).AND.(X(2).GT.0).AND.(IZ.EQ.0))
C IS TRUE ON RETURN FROM MPCMI.
      COMMON /MPCOM/ B, T, DUMMY
      INTEGER I, IB, MXINT, XS, X2,
     $  B, DUMMY(21), IZ, MPGET, MPPARN, T, X(1)
      XS = X(1)
      IZ = 0
C RETURN ZERO IF X ZERO OR EXPONENT(X) .LE. 0.
      IF (XS.EQ.0) RETURN
      X2 = MPGET (X, 2)
      IF (X2.LE.0) RETURN
      MXINT = MPPARN (16)
      IB = MXINT/B
C LOOP TO CONVERT INTEGER PART OF X TO SINGLE-PRECISION INTEGER.
      DO 10 I = 1, X2
C CHECK IF B*IZ WOULD EXCEED MXINT
      IF (IZ .GT. IB) GO TO 20
      IZ = B*IZ
      IF (I .GT. T) GO TO 10
C CHECK IF X(I+2) + IZ WOULD EXCEED MXINT
      IF (X(I+2) .GT. (MXINT - IZ)) GO TO 20
      IZ = IZ + X(I+2)
   10 CONTINUE
C RESTORE SIGN AND RETURN
      IZ = XS*IZ
      RETURN
C HERE ABS(INT(X)) IS LARGER THAN MXINT, SO RETURN ZERO.
   20 IZ = 0
      RETURN
      END
C $$                   ******  MPCMP  ******
      INTEGER FUNCTION MPCMP (X, Y)
C COMPARES THE UNPACKED MULTIPLE-PRECISION NUMBERS X AND Y,
C RETURNING +1 IF X .GT. Y,
C           -1 IF X .LT. Y,
C AND        0 IF X .EQ. Y.
      INTEGER X(1), Y(1), T2, I, MPPARN
C COMPARE SIGNS OF X AND Y.
      IF (X(1) - Y(1)) 10, 30, 20
C X .LT. Y
   10 MPCMP = -1
      RETURN
C X .GT. Y
   20 MPCMP = 1
      RETURN
C SIGN(X) = SIGN(Y), SEE IF ZERO
   30 IF (X(1).EQ.0) GO TO 55
C HAVE TO COMPARE EXPONENTS AND FRACTIONS
      T2 = MPPARN(2) + 2
      DO 50 I = 2, T2
      IF (X(I) - Y(I)) 60, 50, 70
   50 CONTINUE
C NUMBERS EQUAL
   55 MPCMP = 0
      RETURN
C ABS(X) .LT. ABS(Y)  AND SIGNS EQUAL.
   60 MPCMP = -X(1)
      RETURN
C ABS(X) .GT. ABS(Y)  AND SIGNS EQUAL.
   70 MPCMP = X(1)
      RETURN
      END
C $$                   ******  MPCMPI  ******
      INTEGER FUNCTION MPCMPI (X, I)
C COMPARES MP NUMBER X WITH INTEGER I, RETURNING
C     +1 IF X .GT. I,
C      0 IF X .EQ. I,
C     -1 IF X .LT. I
C X MAY BE PACKED OR UNPACKED.
      COMMON R
      INTEGER I, I2, MPCOMP, R(1), SV, X(1)
      CALL MPSAVN (SV)
      CALL MPNEW (I2)
C CONVERT I TO MULTIPLE-PRECISION AND COMPARE
      CALL MPCIM (I, R(I2))
      MPCMPI = MPCOMP (X, R(I2))
      CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPCOMP  ******
      INTEGER FUNCTION MPCOMP (X, Y)
C COMPARES THE MULTIPLE-PRECISION NUMBERS X AND Y,
C RETURNING +1 IF X .GT. Y,
C           -1 IF X .LT. Y,
C AND        0 IF X .EQ. Y.
C X AND Y MAY BE PACKED OR UNPACKED.
      COMMON R
      INTEGER R(1), X(1), Y(1), SV, MPCMP, I2, I3
C FOR EFFICIENCY TREAT UNPACKED NUMBERS SEPARATELY
      IF (MAX0 (IABS(X(1)), IABS(Y(1))) .GT. 1) GO TO 10
      MPCOMP = MPCMP (X, Y)
      RETURN
C HERE X AND/OR Y IS PACKED
   10 CALL MPSAVN (SV)
      CALL MPNEW (I2)
      CALL MPNEW (I3)
      CALL MPUNPK (X, R(I2))
      CALL MPUNPK (Y, R(I3))
      MPCOMP = MPCMP (R(I2), R(I3))
      CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPCQM  ******
      SUBROUTINE MPCQM (I, J, Q)
C CONVERTS THE RATIONAL NUMBER I/J TO MULTIPLE PRECISION Q.
C ROUNDING IS DEFINED BY THE PARAMETER RNDRL IN COMMON /MPCOM/ -
C SEE SUBROUTINE MPNZR.
      INTEGER I, I1, J, J1, Q(1)
      I1 = I
      J1 = J
C REMOVE ANY COMMON FACTOR OF I AND J.
      CALL MPGCD (I1, J1)
C CHECK SIGN OF DENOMINATOR.
      IF (J1) 20, 10, 30
C DIVISION BY ZERO
   10 CALL MPERRM (23HJ = 0 IN CALL TO MPCQM$)
      RETURN
C MAKE DENOMINATOR POSITIVE TO GIVE CORRECT DIRECTED ROUNDING.
   20 I1 = -I1
      J1 = -J1
   30 CALL MPCIM (I1, Q)
      IF (J1.NE.1) CALL MPDIVI (Q, J1, Q)
      RETURN
      END
C $$                   ******  MPDIGS  ******
      INTEGER FUNCTION MPDIGS (N)
C RETURNS THE NUMBER OF MP DIGITS (TO BASE B) REQUIRED FOR THE
C EQUIVALENT OF AT LEAST N FLOATING (BASE OUTBAS) PLACES,
C B**(MPDIGS-1) .GE. OUTBAS**(N-1) .
C IF OUTBAS HAS ITS DEFAULT VALUE OF 10, RESULT IS THE NUMBER
C OF BASE B DIGITS TO GIVE THE EQUIVALENT OF AT LEAST N FLOATING
C DECIMAL PLACES.
      INTEGER MPCHGB, MPPARN, N
      MPDIGS = MAX0 (2, MPCHGB (MPPARN(1), MPPARN(20), N-1) + 1)
      RETURN
      END
C $$                   ******  MPDIGW  ******
      INTEGER FUNCTION MPDIGW (N)
C RETURNS CHARACTER 0, ... , 9, A, ... , F IF N IS
C                   0, ... , 9, 10, ... , 15 RESPECTIVELY,
C RETURNS CHARACTER * OTHERWISE.
      INTEGER D(17), I, N
      DATA D(1), D(2), D(3), D(4) /1H0, 1H1, 1H2, 1H3/
      DATA D(5), D(6), D(7), D(8) /1H4, 1H5, 1H6, 1H7/
      DATA D(9), D(10), D(11), D(12) /1H8, 1H9, 1HA, 1HB/
      DATA D(13), D(14), D(15), D(16) /1HC, 1HD, 1HE, 1HF/
      DATA D(17) /1H*/
      I = MIN0 (N, 16)
      IF (I .LT. 0) I = 16
      MPDIGW = D(I+1)
      RETURN
      END
C $$                   ******  MPDIV  ******
      SUBROUTINE MPDIV (X, Y, Z)
C
C SETS Z = X/Y, FOR MP X, Y AND Z.
C USES MPDIVL FOR SMALL T, MPREC AND MPMUL FOR LARGE T, SO TIME
C IS O(M(T)).  OVER/UNDERFLOW IS DETECTED BY SUBROUTINE MPNZR.
C
C ROUNDING IS DETERMINED BY RNDRL IN COMMON /MPCOM/ AS FOLLOWS -
C RNDRL = 0 - ERROR LESS THAN 1 UNIT IN LAST PLACE (ULP), SO EXACT
C             IF RESULT CAN BE EXACTLY REPRESENTED.
C RNDRL = 1 - SEE MPNZR.
C RNDRL = 2 OR 3 - DIRECTED ROUNDINGS - SEE SUBROUTINE MPNZR.
C
      COMMON R
      COMMON /MPCOM/ B, T, M, DUMMY
      INTEGER CROSS, I2, I3, SV, TG
      INTEGER B, DUMMY(20), M, MPGD, R(1), T, X(1), Y(1), Z(1)
C FOLLOWING CROSSOVER POINT DETERMINED EMPIRICALLY
      DATA CROSS /80/
C SAVE T, M, RNDRL ETC.
      CALL MPSAVN (SV)
C CHECK FOR DIVISION BY ZERO
      IF (Y(1) .EQ. 0) CALL MPERRM (
     $  44HATTEMPTED DIVISION BY ZERO IN CALL TO MPDIV$)
C CHECK FOR X = 0
      IF (X(1).NE.0) GO TO 10
      Z(1) = 0
      GO TO 30
C SEE IF T SMALL ENOUGH THAT MPDIVL IS FASTER THAN MPREC
C OR MUST USE MPDIVL TO ENSURE CORRECT ROUNDED RESULT
   10 TG = T + 1 + MPGD(100)
      IF ((TG.LT.CROSS) .OR. (R(SV+2).NE.0)) GO TO 20
C HERE USE MPREC AND MPMUL.  INCREASE T AND M TEMPORARILY.
      T = TG
      M = M + T
C ALLOCATE TEMPORARY SPACE
      CALL MPNEW (I2)
C MOVE Y AND COMPUTE RECIPROCAL, TAKING CARE FOR DIRECTED ROUNDING
      CALL MPMOVE (Y, IABS(R(SV)), R(I2), TG)
      CALL MPREVR (-X(1))
      CALL MPREC (R(I2), R(I2))
C RESTORE RNDRL AND MOVE X FOR MULTIPLICATION
      CALL MPSETR (IABS(R(SV+2)))
      CALL MPNEW (I3)
      CALL MPMOVE (X, IABS(R(SV)), R(I3), TG)
      CALL MPMUL (R(I3), R(I2), R(I2))
C RESTORE M (SO MPNZR CAN DETECT OVERFLOW/UNDERFLOW)
      M = R(SV+1)
C ROUND RESULT (TO NEAREST IF RNDRL = 0 OR 1)
      CALL MPRND (R(I2), TG, Z, IABS(R(SV)), 0)
      GO TO 30
C HERE FASTER OR NECESSARY TO USE MPDIVL
   20 CALL MPDIVL (X, Y, Z)
C RESTORE EVERYTHING
   30 CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPDIVI  ******
      SUBROUTINE MPDIVI (X, IY, Z)
C DIVIDES MP X BY THE SINGLE-PRECISION INTEGER IY GIVING MP Z.
C THIS IS MUCH FASTER THAN DIVISION BY AN MP NUMBER.
C ROUNDING IS DEFINED BY PARAMETER RNDRL IN COMMON /MPCOM/ -
C SEE COMMENTS IN SUBROUTINE MPNZR.
      COMMON R
      COMMON /MPCOM/ B, T, M, LUN, MXR, SPTR, MXSPTR, MNSPTR,
     $  MXEXPN, MNEXPN, RNDRL, DUMMY
      INTEGER I2, J, KG, MPGD, RE, RS, TWO,
     $  B, DUMMY(12), IY, LUN, M, MNEXPN, MNSPTR, MXEXPN, MXR,
     $  MXSPTR, R(1), RNDRL, SPTR, T, X(1), Z(1)
      TWO = 2
C COMPUTE NUMBER OF GUARD DIGITS REQUIRED.
      KG = MIN0 (1, RNDRL)
      IF (RNDRL.EQ.1) KG = 1 + MPGD (IY)
C ALLOCATE T+KG WORDS FOR ACCUMULATOR IF NECESSARY.
      I2 = SPTR
      IF (KG.GT.0) CALL MPNEW2 (I2, T + KG)
C GET SIGN OF X
      RS = X(1)
      J = IABS(IY)
C CHECK SIGN OF DIVISOR
      IF (IY) 20, 10, 30
   10 CALL MPERRM (45HATTEMPTED DIVISION BY ZERO IN CALL TO MPDIVI$)
C ALLOW FOR NEGATIVE DIVISOR
   20 RS = -RS
C CHECK FOR ZERO DIVIDEND
   30 IF (RS.NE.0) GO TO 40
      Z(1) = 0
      GO TO 100
   40 RE = X(TWO)
C CHECK FOR DIVISION BY +-B (THIS IS COMMON - SEE MPOUT2)
      IF (J.NE.B) GO TO 70
      RE = RE - 1
C MOVE X TO Z.
   50 CALL MPSTR (X, Z)
C FIX UP SIGN AND EXPONENT OF RESULT.
   60 Z(1) = RS
      Z(TWO) = RE
C CHECK FOR UNDERFLOW
      IF (RE.LE.(-M)) GO TO 90
C UPDATE MNEXPN
      MNEXPN = MIN0 (RE, MNEXPN)
      GO TO 100
C CHECK FOR DIVISION BY +-1
   70 IF (J.EQ.1) GO TO 50
C GENUINE DIVISION HERE.  CHECK NUMBER OF GUARD DIGITS.
      IF (KG.GT.0) GO TO 80
C HERE CAN USE Z(3), ... , Z(T+2) AS ACCUMULATOR.
      CALL MPDIV2 (X(TWO+1), J, 0, RE, Z(TWO+1))
      GO TO 60
C HERE USE R(I2), ... , R(SPTR-1) AS ACCUMULATOR
   80 CALL MPDIV2 (X(TWO+1), J, KG, RE, R(I2))
      CALL MPNZR (RS, RE, Z(1), R(I2), KG)
      GO TO 100
C UNDERFLOW HERE
   90 CALL MPUNFL (Z)
C RESTORE STACK POINTER AND RETURN
  100 SPTR = I2
      RETURN
      END
C $$                   ******  MPDIVL  ******
      SUBROUTINE MPDIVL (X, Y, Z)
C DIVIDES X BY Y, GIVING RESULT Z, FOR MP X, Y AND Z.
C USES LONG DIVISION METHOD, TIME O(T**2).   AN ALTERNATIVE METHOD
C (IMPLEMENTED IN MPDIV) WITH TIME O(M(T)) IS TO USE MPREC AND
C MPMUL. WITH THE PRESENT IMPLEMENTATION OF MPMUL, THIS IS FASTER
C (BY UP TO A FACTOR OF ABOUT 2) FOR LARGE T, BUT MPDIVL IS FASTER
C FOR SMALL T (SEE DATA CROSS /.../ IN MPDIV AND MPREC).
C ROUNDING IS DETERMINED BY RNDRL IN COMMON /MPCOM/ -
C SEE SUBROUTINE MPNZR.
      COMMON R
      INTEGER I2, I3, KG, R(1), SV, TWO, X(1), Y(1), Z(1)
      TWO = 2
C CHECK FOR DIVISION BY ZERO.
      IF (Y(1) .EQ. 0) CALL MPERRM (
     $  35HDIVISION BY ZERO IN CALL TO MPDIVL$)
C CHECK FOR X ZERO.
      IF (X(1).NE.0) GO TO 10
      Z(1) = 0
      RETURN
C HERE X AND Y ARE NONZERO.  DETERMINE NUMBER OF GUARD DIGITS.
   10 CALL MPSAVN (SV)
      KG = MIN0 (2, R(SV+2)+1)
      IF (R(SV+2).EQ.1) KG = R(SV) + 2
C ALLOCATE TEMPORARY STORAGE
      CALL MPNEW2 (I2, R(SV) + KG)
      CALL MPNEW (I3)
C MOVE X TO TEMPORARY STORAGE AS MPDIV3 OVERWRITES FIRST ARGUMENT.
      CALL MPSTR (X, R(I3))
C DO LONG DIVISION
      CALL MPDIV3 (R(I3+1), Y(TWO), R(I2), KG)
C NORMALIZE AND ROUND RESULT
      CALL MPNZR (X(1)*Y(1), X(TWO)+1-Y(TWO), Z, R(I2), KG)
C RESTORE STACK POINTER AND RETURN.
      CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPDIV2  ******
      SUBROUTINE MPDIV2 (X, J, KG, RE, A)
C CALLED BY MPDIVI, NOT FOR INDEPENDENT USE.
C ASSUMES X(1) ... X(T) REPRESENTS A BASE B NUMBER, X(1) .GT. 0,
C J .GT. 0.   DIVIDES BY J, RESULT IN A(1) ... A(T+KG).  THE RESULT
C IS LEFT SHIFTED SO A(1) .GT. 0 AND RE DECREMENTED ACCORDINGLY.
C ASSUMES KG. GE. 0, ALSO KG .GT. 0 IF RNDRL .GT. 0 .
      COMMON /MPCOM/ B, T, M, LUN, MXR, SPTR, MXSPTR, MNSPTR, MXEXPN,
     $  MNEXPN, RNDRL, KTUNFL, MXUNFL, DECPL, MT2, MXINT, DUMMY
      INTEGER B2, C, C2, I, IQ, IQJ, IR, J1, J11, J2, K, KH, KHH, R1,
     $  A(1), B, DECPL, DUMMY(7), J, KG, KTUNFL, LUN, M, MNEXPN,
     $  MNSPTR, MT2, MXEXPN, MXINT, MXR, MXSPTR, MXUNFL, RE, RNDRL,
     $  SPTR, T, X(1)
      C = 0
      I = 0
      KHH = T + KG
C IF J*B NOT REPRESENTABLE AS AN INTEGER HAVE TO SIMULATE
C LONG DIVISION.
      B2 = MXINT/B
      IF (J.GE.B2) GO TO 70
C LOOK FOR FIRST NONZERO DIGIT IN QUOTIENT
   10 I = I + 1
      C = B*C
      IF (I.LE.T) C = C + X(I)
      R1 = C/J
      IF (R1) 150, 10, 20
C ADJUST EXPONENT AND GET T+KG DIGITS IN QUOTIENT
   20 RE = RE + 1 - I
      A(1) = R1
      C = B*(C - J*R1)
      KH = 2
      IF (I.GE.T) GO TO 40
      KH = T + 1 - I
      DO 30 K = 2, KH
      I = I + 1
      C = C + X(I)
      A(K) = C/J
   30 C = B*(C - J*A(K))
      IF (C.LT.0) GO TO 150
      KH = KH + 1
   40 IF (KH.GT.KHH) GO TO 60
      DO 50 K = KH, KHH
      A(K) = C/J
   50 C = B*(C - J*A(K))
      IF (C.LT.0) GO TO 150
C ADJUST LAST DIGIT FOR DIRECTED ROUNDING
   60 IF ((RNDRL.GT.1).AND.(C.GT.0)) A(KHH) = 1
      RETURN
C HERE NEED SIMULATED DOUBLE-PRECISION DIVISION
   70 C2 = 0
      J1 = J/B
      J2 = J - J1*B
      J11 = J1 + 1
C LOOK FOR FIRST NONZERO DIGIT
   80 I = I + 1
      C = B*C + C2
      C2 = 0
      IF (I.LE.T) C2 = X(I)
      IF (C-J1)  80, 90, 100
   90 IF (C2.LT.J2) GO TO 80
C COMPUTE T+KG QUOTIENT DIGITS
  100 RE = RE + 1 - I
      K = 1
      GO TO 120
C MAIN LOOP FOR LARGE ABS(IY) CASE
  110 K = K + 1
      IF (K.GT.KHH) GO TO 60
      I = I + 1
C GET APPROXIMATE QUOTIENT FIRST
  120 IR = C/J11
C NOW REDUCE SO OVERFLOW DOES NOT OCCUR
      IQ = C - IR*J1
      IF (IQ.LT.B2) GO TO 130
C HERE IQ*B WOULD POSSIBLY OVERFLOW SO INCREASE IR
      IR = IR + 1
      IQ = IQ - J1
  130 IQ = IQ*B - IR*J2
      IF (IQ.GE.0) GO TO 140
C HERE IQ NEGATIVE SO IR WAS TOO LARGE
      IR = IR - 1
      IQ = IQ + J
  140 IF (I.LE.T) IQ = IQ + X(I)
      IQJ = IQ/J
C A(K) = QUOTIENT, C = REMAINDER
      A(K) = IQJ + IR
      C = IQ - J*IQJ
      IF (C.GE.0) GO TO 110
C CARRY NEGATIVE SO OVERFLOW MUST HAVE OCCURRED
  150 CALL MPERRM (40HINTEGER OVERFLOW IN MPDIV2, B TOO LARGE$)
      RETURN
      END
C $$                   ******  MPDIV3  ******
      SUBROUTINE MPDIV3 (X, Y, D, KG)
C CALLED BY MPDIVL.  ASSUMES X(2), ... X(T+1) AND
C Y(2), ... , Y(T+1) CONTAIN BASE B INTEGERS, X(2) .GT. 0,
C Y(2) .GT. 0, KG .GE. 0.  ALSO ASSUMES KG .GE. 2 IF RNDRL .GT. 0.
C RETURNS FIRST T+KG DIGITS OF QUOTIENT X/Y IN D(1), ... , D(T+KG).
C DESTROYS X(1), ... , X(T+1), USES BUT RESTORES Y(1).
C ARGUMENTS MUST NOT OVERLAP.
      COMMON /MPCOM/ B, T, DUMMY
      INTEGER B1, C, I, IR, J, K, Q, TG, TWO, T1, T2, Y1S,
     $  B, D(1), DUMMY(21), KG, MPPARN, T, X(1), Y(1)
      TWO = 2
C SAVE Y(1) AND SET TO ZERO.
      Y1S = Y(1)
      Y(1) = 0
C SET UP OUTER LOOP.
      X(1) = 0
      B1 = B*(B-1)
      T1 = T + 1
      T2 = T1 + 1
      TG = T + KG
C OUTER LOOP
      DO 70 J = 1, TG
      D(J) = 0
C COMPARE X AND Y (USUALLY ONLY ONCE FOR EACH J).
   10 DO 20 I = 1, T1
      IF (X(I) - Y(I)) 50, 20, 30
   20 CONTINUE
C HERE B*Y .GT. X .GE. Y,   COMPUTE LOWER BOUND ON QUOTIENT X/Y.
   30 Q = MAX0 (1, (B*X(1) + X(TWO))/(Y(TWO) + 1))
C COMPUTE SHARPER LOWER BOUND IF OVERFLOW IMPOSSIBLE.
      IF ((X(1)+1) .LT. (MPPARN(16)/(B*B))) Q =
     $  MAX0 (Q, (B*(B*X(1)+X(TWO))+X(TWO+1))/
     $  (B*Y(TWO)+Y(TWO+1)+1))
C INCREMENT J-TH QUOTIENT DIGIT
      D(J) = D(J) + Q
      I = T2
C K IS (B + SIGNED CARRY), 0 .LT. K .LE. B.
      K = B
C INNER LOOP
      DO 40 IR = 1, T1
      I = I - 1
      C = B1 + K + X(I) - Q*Y(I)
      K = C/B
   40 X(I) = C - B*K
C THERE SHOULD BE NO CARRY OFF END.  CHECK JUST IN CASE.
      IF (K.EQ.B) GO TO 10
      CALL MPERRM (36HINTEGER OVERFLOW OCCURRED IN MPDIV3$)
C SHIFT X LEFT
   50 DO 60 I = 1, T
   60 X(I) = X(I+1)
C LAST STATEMENT OF OUTER LOOP.
   70 X(T1) = 0
C RESTORE Y(1).
      Y(1) = Y1S
C RETURN IF RNDRL .LE. 1
      IF (MPPARN(11).LE.1) RETURN
C IF REMAINDER NONZERO MAKE SURE A GUARD DIGIT NONZERO FOR
C DIRECTED ROUNDING (ASSUMES KG .GE. 2 HERE)
      DO 80 I = 1, T
   80 D(TG) = MAX0 (D(TG), X(I))
      RETURN
      END
C $$                   ******  MPERR  ******
      SUBROUTINE MPERR
C THIS ROUTINE IS CALLED WHEN A FATAL ERROR CONDITION IS ENCOUNTERED,
C AND AFTER A MESSAGE HAS BEEN WRITTEN ON LOGICAL UNIT LUN.
      COMMON /MPCOM/ SOME, LUN, REST
      INTEGER SOME(3), LUN, REST(19)
      WRITE (LUN, 10) SOME, REST
   10 FORMAT (42H *** EXECUTION TERMINATED BY CALL TO MPERR,
     $        25H IN MP VERSION 810614 ***/
     $        14H *** B, T, M =, I22, 2I20/
     $        24H *** MXR, SPTR, MXSPTR =, I12, 2I20/
     $        29H *** MNSPTR, MXEXPN, MNEXPN =, I7, 2I20/
     $        28H *** RNDRL, KTUNFL, MXUNFL =, I8, 2I20/
     $        24H *** DECPL, MT2, MXINT =, I12, 2I20/
     $        36H *** EXWID, INRECL, INBASE, OUTBAS =, 4I4/
     $        28H *** EXPCH, CHWORD, ONESCP =, 11X, A1, 2I4//)
C ANSI VERSION USES STOP, UNIVAC 1108 VERSION USES
C RETURN 0 IN ORDER TO GIVE A TRACE-BACK.
      IF (.TRUE.) STOP
      RETURN
      END
C $$                   ******  MPERRM  ******
      SUBROUTINE MPERRM (A)
C THE ARGUMENT A IS A HOLLERITH STRING TERMINATED BY $.
C MPERRM WRITES THE STRING OUT ON UNIT LUN, THEN CALLS MPERR.
C ONLY THE FIRST 71 CHARACTERS OF A ARE SIGNIFICANT, AND
C ARE PRECEEDED AND FOLLOWED BY THREE ASTERISKS.
      COMMON R
      COMMON /MPCOM/ B, T, M, LUN, DUMMY
      INTEGER A(1), I, I2, J, LEN, R(1), SV
      INTEGER B, T, M, LUN, DUMMY (19)
      LOGICAL ERR
      CALL MPCHK
      CALL MPSAVN (SV)
C GET SPACE FOR UNPACKING A
      CALL MPNEW2 (I2, 80)
C UNPACK A TO FIND ITS LENGTH AND ALLOW PRINTING IN A1 FORMAT.
      CALL MPUPK (A, R(I2+5), 71, LEN)
C INSERT LEADING AND TRAILING ASTERISKS.
      CALL MPUPK (5H *** , R(I2), 5, J)
      I = I2 + LEN
      CALL MPUPK (4H ***, R(I+5), 4, J)
C WRITE USING MPIO.
      CALL MPIO (R(I2), LEN+9, LUN, 6H(80A1), ERR)
C AND TERMINATE EXECUTION BY CALLING MPERR.
C (DO NOT CALL MPRESN BECAUSE POSSIBLE RECURSION.)
      CALL MPERR
      RETURN
      END
C $$                   ******  MPEXP  ******
      SUBROUTINE MPEXP (X, Y)
C RETURNS Y = EXP(X) FOR PACKED OR UNPACKED MP X, UNPACKED MP Y.
C TIME IS O(SQRT(T)M(T)).
C ROUNDING IS DEFINED BY THE PARAMETER RNDRL IN COMMON /MPCOM/ -
C RNDRL = 0 OR 1 - ERROR LESS THAN 0.6 UNITS IN LAST PLACE.
C RNDRL = 2 OR 3 - SEE COMMENTS IN SUBROUTINE MPNZR.
      COMMON R
      COMMON /MPCOM/ B, T, M, LUN, DUMMY
      INTEGER I2, K, RP, SV, TG
      INTEGER B, DUMMY(19), LUN, M, R(1), T, X(1), Y(1)
      INTEGER MPGD, MPGET
C SAVE T ETC.
      CALL MPSAVN (SV)
C CHECK FOR X = 0
      IF (X(1).NE.0) GO TO 10
      CALL MPCIM (1, Y)
      GO TO 50
C CHECK IF ABS(X) .LT. 1
   10 IF (MPGET (X, 2) .GT. 0) GO TO 20
C USE MPEXP1 HERE
      T = T + MPGD (100)
      TG = T
      CALL MPNEW (I2)
      CALL MPMOVE (X, IABS(R(SV)), R(I2), TG)
      CALL MPEXP1 (R(I2), R(I2))
      CALL MPADDI (R(I2), 1, R(I2))
      GO TO 40
C INCREASE T AS NECESSARY.
   20 RP = MAX0 (MPGET (X, 2), 0)
      T = T + 1 + RP + MPGD (100)
      TG = T
C USE TRUNCATION RATHER THAN ROUNDING INTERNALLY
      IF (R(SV+2) .EQ. 1) CALL MPSETR (0)
      CALL MPNEW (I2)
C MOVE X AND DIVIDE BY B**RP
      CALL MPMOVE (X, IABS(R(SV)), R(I2), TG)
      R(I2+1) = R(I2+1) - RP
C USE EXP1 TO COMPUTE EXP(...) - 1
      CALL MPEXP1 (R(I2), R(I2))
      CALL MPADDI (R(I2), 1, R(I2))
C SEE IF RESULT ALREADY OBTAINED (RP .LE. 0)
      IF (RP.LE.0) GO TO 40
C HERE HAVE TO RAISE RESULT TO B-TH POWER RP TIMES
C UNDERFLOW/OVERFLOW WILL OCCUR HERE IF X OUT OF ALLOWABLE
C RANGE.
      DO 30 K = 1, RP
   30 CALL MPPWRA (R(I2), IABS(B), R(I2))
C ROUND RESULT TO TS DIGITS, RESTORE T ETC. AND RETURN
   40 CALL MPRND (R(I2), TG, Y, IABS(R(SV)), 0)
   50 CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPEXP1  ******
      SUBROUTINE MPEXP1 (X, Y)
C ASSUMES THAT X AND Y ARE MP NUMBERS,  -1 .LT. X .LT. 1.
C X MAY BE PACKED OR UNPACKED, Y IS UNPACKED.
C RETURNS Y = EXP(X) - 1 USING AN O(SQRT(T).M(T)) ALGORITHM
C DESCRIBED IN - R. P. BRENT, THE COMPLEXITY OF MULTIPLE-
C PRECISION ARITHMETIC (IN COMPLEXITY OF COMPUTATIONAL PROBLEM
C SOLVING, UNIV. OF QUEENSLAND PRESS, BRISBANE, 1976, 126-165).
C ASYMPTOTICALLY FASTER METHODS EXIST, BUT ARE NOT USEFUL
C UNLESS T IS VERY LARGE. SEE COMMENTS TO MPATAN AND MPPIGL.
C ROUNDING IS DEFINED BY THE PARAMETER RNDRL IN COMMON /MPCOM/ -
C RNDRL = 0 OR 1 - ERROR LESS THAN 0.6 UNITS IN LAST PLACE.
C RNDRL = 2 OR 3 - SEE COMMENTS IN SUBROUTINE MPNZR.
      COMMON R
      COMMON /MPCOM/ B, T, M, LUN, MXR, SPTR, MXSPTR, MNSPTR,
     $  MXEXPN, MNEXPN, RNDRL, KTUNFL, DUMMY
      INTEGER I, I2, I3, I4, KTU, LA, MPGET, MPTLB, Q, SV, TG,
     $  B, DUMMY(11), KTUNFL, LUN, M, MNEXPN, MNSPTR, MXEXPN, MXR,
     $  MXSPTR, R(1), RNDRL, SPTR, T, X(1), Y(1),
     $  LIM, MPPARN, MUL
C CHECK FOR X = 0
      Y(1) = X(1)
      IF (X(1) .EQ. 0) RETURN
C SAVE T, M ETC.
      CALL MPSAVN (SV)
C CHECK THAT ABS(X) .LT. 1
      IF (MPGET (X, 2) .GT. 0) CALL MPERRM (
     $  41HABS(X) NOT LESS THAN 1 IN CALL TO MPEXP1$)
C COMPUTE APPROXIMATELY OPTIMAL Q.
      Q = 0
      IF (MPGET (X, 2) .LT. (-T)) GO TO 30
      LA = MPTLB (T/3)
C INTEGER SQUARE ROOT APPROXIMATION HERE
   20 Q = Q + 1
      IF ((Q*Q) .LE. LA) GO TO 20
      Q = MAX0 (0, Q + MPGET (X, 2)*MPTLB(1))
C INCREASE T AS NECESSARY.
   30 CALL MPGD3 (Q, TG)
C ALLOCATE TEMPORARY SPACE
   40 CALL MPNEW (I2)
      CALL MPNEW (I3)
      CALL MPNEW (I4)
C USE TRUNCATION RATHER THAN ROUNDING INTERNALLY.
      IF (RNDRL.EQ.1) RNDRL = 0
C INCREASE M (TO AVOID UNDERFLOW PROBLEMS) AND MOVE X
      M = M + 2*T
      CALL MPMOVE (X, IABS(R(SV)), R(I2), TG)
      KTU = KTUNFL
C HALVE Q TIMES
C CAN NOT USE MPSCAL HERE AS POTENTIALLY RECURSIVE
      IF (Q .LE. 0) GO TO 60
      LIM = MPPARN(16)/2
      MUL = 1
      DO 50 I = 1, Q
      MUL = 2*MUL
      IF ((MUL .LT. LIM) .AND. (MUL .NE. B) .AND.
     $  (I .NE. Q)) GO TO 50
      CALL MPDIVI (R(I2), MUL, R(I2))
      MUL = 1
   50 CONTINUE
C IF UNDERFLOW OCCURED SET Q = 0 AND TRY AGAIN WITH NEW T ETC.
      IF (KTU .EQ. KTUNFL) GO TO 60
      Q = 0
      CALL MPRESN (SV)
      CALL MPSAVN (SV)
      CALL MPGD3 (MPTLB(IABS(T)), TG)
      GO TO 40
   60 CALL MPSTR (R(I2), R(I4))
      CALL MPSTR (R(I2), R(I3))
      I = 1
      TG = T
C SUM SERIES, REDUCING T WHERE POSSIBLE
   70 T = TG + 2 + R(I3+1) - R(I4+1)
      IF (T.LE.2) GO TO 80
      T = MIN0 (T, TG)
      IF (RNDRL.NE.0) T = TG
      CALL MPMUL (R(I2), R(I3), R(I3))
      I = I + 1
      CALL MPDIVI (R(I3), I, R(I3))
      T = TG
      CALL MPADD (R(I3), R(I4), R(I4))
C CAN ONLY FALL THROUGH NEXT STATEMENT IF UNDERFLOW OCCURRED.
      IF (KTU.EQ.KTUNFL) GO TO 70
C RESTORE WORKING PRECISION
   80 T = TG
C CHECK IF ROUNDING UP REQUIRED
      IF (RNDRL.NE.3) GO TO 90
C ROUNDING UP HERE, SO ADD ABS(LAST TERM)
      R(I3) = IABS(R(I3))
      CALL MPADD (R(I3), R(I4), R(I4))
C FINISHED IF Q .LE. 0
   90 IF (Q.LE.0) GO TO 110
C APPLY (X+1)**2 - 1 = X(2 + X) FOR Q ITERATIONS
      DO 100 I = 1, Q
      CALL MPADDI (R(I4), 2, R(I2))
  100 CALL MPMUL (R(I2), R(I4), R(I4))
C ROUND RESULT
  110 CALL MPRND (R(I4), TG, Y, IABS(R(SV)), 0)
C RESTORE STACK POINTER ETC AND RETURN
      CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPGCD  ******
      SUBROUTINE MPGCD (K, L)
C RETURNS K = K/GCD AND L = L/GCD, WHERE GCD IS THE
C GREATEST COMMON DIVISOR OF K AND L.
C SAVE INPUT PARAMETERS IN LOCAL VARIABLES
      INTEGER I, IS, J, JS, K, L
      I = K
      J = L
      IS = IABS(I)
      JS = IABS(J)
      IF (JS.EQ.0) GO TO 30
C EUCLIDEAN ALGORITHM LOOP
   10 IS = MOD (IS, JS)
      IF (IS.EQ.0) GO TO 20
      JS = MOD (JS, IS)
      IF (JS.NE.0) GO TO 10
      JS = IS
C HERE JS IS THE GCD OF I AND J
   20 K = I/JS
      L = J/JS
      RETURN
C IF J = 0 RETURN (1, 0) UNLESS I = 0, THEN (0, 0)
   30 K = 1
      IF (IS.EQ.0) K = 0
      L = 0
      RETURN
      END
C $$                   ******  MPGD  ******
      INTEGER FUNCTION MPGD (N)
C RETURNS CEILING (LN (MAX (1, ABS(N))) / LN(B)),
C I.E. THE MINIMUM VALUE OF J .GE. 0 SUCH THAT
C     B**J .GE. ABS(N) .
C THIS FUNCTION IS USEFUL FOR COMPUTING THE NUMBER OF GUARD DIGITS
C REQUIRED FOR VARIOUS MULTIPLE-PRECISION CALCULATIONS.
      COMMON /MPCOM/ B, DUMMY
      INTEGER B, DUMMY(22), J, K, N
C CHECK THAT B IS LEGAL.
      IF (B.LT.2) CALL MPCHK
C SET UP LOOP
      J = 0
      K = IABS(N) - 1
      GO TO 20
C LOOP TO COMPUTE RESULT
   10 J = J + 1
      K = K/B
   20 IF (K.GT.0) GO TO 10
C RETURN RESULT J
      MPGD = J
      RETURN
      END
C $$                   ******  MPGD3  ******
      SUBROUTINE MPGD3 (N, TG)
C SETS T = T + 1 + MPGD (100*N) IF SAFE TO CALL
C MPGD, EFFECTIVELY ALMOST THE SAME IF NOT.
C ALSO SETS TG = NEW VALUE OF T.
      COMMON /MPCOM/ B, T, DUMMY
      INTEGER B, DUMMY(21), I, MPGD, MPPARN, N, T, TG
C CHECK WHETHER SAFE TO FORM 100*N
      IF (IABS(N) .GE. (MPPARN(16)/100)) GO TO 10
C HERE IT IS
      I = MPGD (100*N)
      GO TO 20
C HERE IT IS NOT, SO CALL MPGD TWICE
   10 I = MPGD (N) + MPGD (100)
C SET T AND TG
   20 T = T + I + 1
      TG = T
      RETURN
      END
C $$                   ******  MPGET  ******
      INTEGER FUNCTION MPGET (X, N)
C RETURNS X(N).  NECESSARY TO AVOID SOME COMPILER DIAGNOSTICS.
C ALSO USEFUL TO AVOID PFORT VERIFIER UNSAFE REFERENCE WARNINGS.
      INTEGER N, X(1)
      MPGET = X(N)
      RETURN
      END
C $$                   ******  MPIO  ******
      SUBROUTINE MPIO (C, N, UNIT, IFORM, ERR)
C
C IF UNIT .GT. 0 WRITES C(1), ... , C(N) IN FORMAT IFORM
C IF UNIT .LE. 0 READS  C(1), ... , C(N) IN FORMAT IFORM
C IN BOTH CASES USES LOGICAL UNIT IABS(UNIT).
C IF IFORM = 1HU THEN UNFORMATTED I/O IS PERFORMED.
C
C ERR IS RETURNED AS TRUE IF N NON-POSITIVE, OTHERWISE FALSE.
C WE WOULD LIKE TO RETURN ERR AS TRUE IF READ/WRITE ERROR DETECTED,
C BUT THIS CAN NOT BE DONE WITH ANSI STANDARD FORTRAN (1966).
C
C UNIVAC ASCII FORTRAN (FTN 5R1AE) DOES NOT WORK IF IFORM
C IS DECLARED WITH DIMENSION 1.  MOST FORTRANS DO THOUGH.
C
C     INTEGER N (UNIVAC FORTRAN V DOES NOT ALLOW)
      INTEGER C(N), IU, U, UNIT
      CHARACTER*1 IFORM(N)
      LOGICAL ERR, MPIS
      DATA U /1HU/
      ERR = (N.LE.0)
      IF (ERR) RETURN
      IU = IABS(UNIT)
      IF (MPIS (IFORM(1), U)) GO TO 10
      IF (UNIT .GT. 0) WRITE (IU, IFORM) C
      IF (UNIT .LE. 0) READ  (IU, IFORM) C
      RETURN
C UNFORMATTED I/O HERE
   10 IF (UNIT .GT. 0) WRITE (IU) C
      IF (UNIT .LE. 0) READ (IU) C
      RETURN
      END
C $$                   ******  MPIS  ******
      LOGICAL FUNCTION MPIS (J, K)
C
C ASSUMES THAT J AND K ARE WORDS CONTAINING ONE CHARACTER IN LEFT-MOST
C POSITION WITH BLANK FILL (READ UNDER A1 FORMAT OR SET BY 1HX DATA
C STATEMENT OR UNPACKED BY MPUPK).  RETURNS .TRUE. IF AND ONLY IF THE
C CHARACTERS ARE EQUAL.  ON SOME MACHINES IT MAY BE NECESSARY TO
C MASK OFF OTHER CHARACTERS OR SHIFT RIGHT BEFORE COMPARISON.
C
C ON SOME MACHINES CHANGE INTEGER TO REAL IN NEXT LINE.
      INTEGER J, K
C
C ON BURROUGHS B6700 REPLACE .EQ. BY .IS. IN NEXT LINE.
      MPIS = (J.EQ.K)
C
      RETURN
      END
C $$                   ******  MPLARG  ******
      SUBROUTINE MPLARG (MXINT)
C
C              *************************    (SEE THE MP
C              *** MACHINE DEPENDENT ***     USERS GUIDE FOR
C              *************************     CONVERSION HINTS)
C
C RETURNS MXINT .LE. THE MAXIMUM REPRESENTABLE INTEGER OF
C THE FORM 2**J - 1 .
C
C INTEGER ARITHMETIC MUST BE PERFORMED EXACTLY ON INTEGERS
C IN THE RANGE -MXINT ... +MXINT, SO ON SOME MACHINES
C MXINT MUST BE LESS THAN THE LARGEST REPRESENTABLE INTEGER,
C E.G. THIS APPLIES ON BURROUGHS B6700 AND CYBER 76 SERIES.
C
      LOGICAL CYBER, B6700, PDP11
      INTEGER I, K, MX, MXINT, WDLEN
C
C WDLEN MAY BE SET TO THE WORDLENGTH (IN BITS) IN THE FOLLOWING
C DATA STATEMENT IF THE CODE BELOW FOR COMPUTING MXINT DOES
C NOT WORK.
C
      DATA WDLEN /0/
C
C ON CYBER 76, BURROUGHS 6700, OR PDP 11 MACHINES SET
C THE CORRESPONDING LOGICAL VARIABLE TO .TRUE.
C
      DATA CYBER /.FALSE./
      DATA B6700 /.FALSE./
      DATA PDP11 /.FALSE./
C
C THE FOLLOWING ASSUMES AT LEAST 2 CHARACTERS PER WORD,
C CHARACTERS 0 AND 1 HAVE CONSECUTIVE BINARY CODES, AND
C LEFTMOST CHARACTER IS IN MOST SIGNIFICANT POSITION.
C IF THESE ASSUMPTIONS FAIL, WDLEN (OR ONE OF CYBER, B6700,
C PDP11) MUST BE SET ABOVE.
C
C FOR A LESS RESTRICTIVE ALGORITHM SEE - ALGORITHMS TO REVEAL
C THE REPRESENTATION OF CHARACTERS, INTEGERS AND FLOATING-POINT
C NUMBERS (BY J. E. GEORGE), ACM TRANS. MATH. SOFTWARE 1(1975),
C 210-216.  UNLIKE GEORGE, WE AVOID THE USE OF AN EXTERNAL UNIT.
C
      K = WDLEN
      IF (CYBER) K = 49
      IF (B6700) K = 40
      IF (PDP11) K = 16
      IF (K .GT. 0) GO TO 10
C HERE TRY TO COMPUTE LOWER BOUND ON MXINT.
      CALL MPSUBA (2H10, 2H00, MXINT)
      CALL MPSUBA (2H01, 2H00, MX)
      MXINT = 2*(MXINT*(MXINT/(4*MX)) - 1) + 1
      RETURN
C HERE K IS LOWER BOUND ON WORDLENGTH, COMPUTE 2**(K-1) - 1
   10 MXINT = 0
      DO 20 I = 2, K
   20 MXINT = 2*MXINT + 1
      RETURN
      END
C $$                   ******  MPLN  ******
      SUBROUTINE MPLN (X, Y)
C RETURNS Y = LN(X), FOR MP X AND Y, USING MPLNI AND MPLNS.
C X MAY BE PACKED OR UNPACKED, Y IS UNPACKED.
C RESTRICTION - INTEGER PART OF LN(X) MUST BE REPRESENTABLE AS A
C SINGLE-PRECISION INTEGER.  TIME IS O(SQRT(T).M(T) + T**2).
C FOR SMALL INTEGER X, MPLNI IS FASTER.
C ASYMPTOTICALLY FASTER METHODS EXIST (EG THE GAUSS-SALAMIN
C METHOD, SEE MPLNGS), BUT ARE NOT USEFUL UNLESS T IS LARGE.
C SEE COMMENTS TO MPATAN, MPEXP1 AND MPPIGL.
C ROUNDING IS DEFINED BY THE PARAMETER RNDRL IN COMMON /MPCOM/ -
C RNDRL = 0 OR 1 - ERROR LESS THAN 0.6 UNITS IN LAST PLACE.
C RNDRL = 2 OR 3 - SEE COMMENTS IN SUBROUTINE MPNZR.
      COMMON R
      COMMON /MPCOM/ B, DUMMY
      INTEGER I2, I3, J, K, SV, TG
      INTEGER B, DUMMY(22), R(1), X(1), Y(1)
C SAVE T ETC.
      CALL MPSAVN (SV)
C INCREASE WORKING PRECISION.
      CALL MPGD3 (1, TG)
C ALLOCATE TEMPORARY SPACE
      CALL MPNEW (I2)
      CALL MPNEW (I3)
C USE TRUNCATED ARITHMETIC.
      CALL MPSETR (0)
C CHECK THAT X IS POSITIVE
      IF (X(1) .LE. 0) CALL MPERRM (
     $  30HX NONPOSITIVE IN CALL TO MPLN$)
C MOVE X TO LOCAL STORAGE
      CALL MPMOVE (X, IABS(R(SV)), R(I2), TG)
C CHECK IF X IS CLOSE TO 1
      CALL MPADDI (R(I2), -1, R(I3))
      IF (R(I3) .EQ. 0) GO TO 10
      IF (R(I3+1) .GE. 0) GO TO 20
C HERE X CLOSE TO 1, SO USE MPLNS DIRECTLY
   10 CALL MPLNS (R(I3), R(I2))
      GO TO 30
C TAKE OFF EXPONENT AND FIRST TWO DIGITS
   20 J = B*R(I2+2) + R(I2+3)
      K = R(I2+1) - 2
      R(I2+1) = 2
      CALL MPDIVI (R(I2), J, R(I2))
      CALL MPADDI (R(I2), -1, R(I2))
C NOW SHOULD BE SAFE TO CALL MPLNS
      CALL MPLNS (R(I2), R(I2))
C NOW USE MPLNI TO CORRECT RESULT
      CALL MPLNI (IABS(B), R(I3))
      CALL MPMULI (R(I3), K, R(I3))
      CALL MPADD (R(I2), R(I3), R(I2))
      CALL MPLNI (J, R(I3))
      CALL MPADD (R(I2), R(I3), R(I2))
C ROUND RESULT
   30 CALL MPSETR (IABS(R(SV+2)))
      CALL MPRND (R(I2), TG, Y, IABS(R(SV)), 1)
C RESTORE T ETC. AND RETURN.
      CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPLNI  ******
      SUBROUTINE MPLNI (N, X)
C RETURNS MULTIPLE-PRECISION X = LN(N) FOR SMALL POSITIVE
C INTEGER N (2*N MUST BE REPRESENTABLE).  TIME IS O(T**2).
C METHOD IS TO USE A RAPIDLY CONVERGING SERIES AND MPL235.
C ROUNDING IS DEFINED BY THE PARAMETER RNDRL IN COMMON /MPCOM/ -
C RNDRL = 0 OR 1 - ERROR LESS THAN 0.6 UNITS IN LAST PLACE.
C RNDRL = 2 OR 3 - SEE COMMENTS IN SUBROUTINE MPNZR.
      COMMON R
      COMMON /MPCOM/ B, T, DUMMY
      INTEGER I, IA, IK, IM, IP, IP2, IQ, IQ2, I2, I3, I4, J, MXINT, N1,
     $  N2, SV, TG, B, DUMMY(21), MPPARN, MPTLB, N, R(1), T, X(1)
C CHECK FOR N = 1 AND N .LT. 1
      IF (N-1) 10, 20, 30
   10 CALL MPERRM (32HN NOT POSITIVE IN CALL TO MPLNI$)
C LN(1) = 0
   20 X(1) = 0
      RETURN
C HERE N .GE. 2, INCREASE T AS NECESSARY.
   30 CALL MPSAVN (SV)
      CALL MPGD3 (MPTLB (IABS(T)), TG)
C USE TRUNCATED ARITHMETIC INTERNALLY
      CALL MPSETR (0)
      MXINT = MPPARN (16)
C ALLOCATE WORKING SPACE
      CALL MPNEW (I2)
      IF (N.GT.2) GO TO 40
C N = 2 IS A SPECIAL CASE
      CALL MPL235 (1, 0, 0, R(I2))
      GO TO 150
C HERE N .GE. 3
   40 IF (N .GT. (MXINT/2)) GO TO 80
      J = 3
      IA = 0
      N2 = N/2
   50 IF (J.GT.N2) GO TO 60
      IA = IA + 1
      J = 2*J
      GO TO 50
C NOW J = 3*(2**IA) .LE. N .LT. 6*(2**IA)
   60 J = J/3
      IM = N
      IK = 0
      DO 70 I = 3, 6
      N1 = I*J
      IF (IABS(N1-N).GT.IM) GO TO 70
      IM = IABS(N1-N)
      IK = I
   70 CONTINUE
      N1 = IK*J
C NOW N IS CLOSE TO N1 = IK*(2**IA)
C AND IK = 3, 4, 5 OR 6, SO MPL235 GIVES LN(N1).
      IF (IK.EQ.3) CALL MPL235 (IA, 1, 0, R(I2))
      IF (IK.EQ.4) CALL MPL235 (IA+2, 0, 0, R(I2))
      IF (IK.EQ.5) CALL MPL235 (IA, 0, 1, R(I2))
      IF (IK.EQ.6) CALL MPL235 (IA+1, 1, 0, R(I2))
      IF (N.EQ.N1) GO TO 150
C NOW NEED LN(N/N1).
      N2 = N
      CALL MPGCD (N2, N1)
      IP = N2 - N1
      IQ = N2 + N1
C CHECK FOR POSSIBLE INTEGER OVERFLOW
      IF (IQ.GT.14) GO TO 100
   80 CALL MPERRM (29HN TOO LARGE IN CALL TO MPLNI$)
C REDUCE TO LOWEST TERMS
  100 CALL MPGCD (IP, IQ)
      CALL MPNEW (I3)
      CALL MPNEW (I4)
      CALL MPCQM (2*IP, IQ, R(I4))
      CALL MPSTR (R(I4), R(I3))
      CALL MPADD (R(I2), R(I3), R(I2))
      I = 1
      IF (IQ.GT.(MXINT/IQ)) GO TO 110
      IQ2 = IQ**2
      IP2 = IP**2
C LOOP TO SUM SERIES FOR LN(N2/N1)
  110 I = I + 2
      IF (R(I4).EQ.0) GO TO 140
C REDUCE T IF POSSIBLE, DONE IF CAN REDUCE BELOW 2
      T = TG + R(I4+1) + 2
      IF (T.LE.2) GO TO 140
      T = MIN0 (T, TG)
C SPLIT UP CALL TO MPMULQ IF IQ TOO LARGE
      IF (IQ.GT.(MXINT/IQ)) GO TO 120
      CALL MPMULQ (R(I4), IP2, IQ2, R(I4))
      GO TO 130
C HERE IQ TOO LARGE FOR ONE CALL TO MPMULQ
  120 CALL MPMULQ (R(I4), IP, IQ, R(I4))
      CALL MPMULQ (R(I4), IP, IQ, R(I4))
  130 CALL MPDIVI (R(I4), I, R(I3))
C RESTORE T AND ACCUMULATE SUM
      T = TG
      CALL MPADD (R(I3), R(I2), R(I2))
      GO TO 110
  140 T = TG
C RESTORE RNDRL AND ROUND RESULT
  150 CALL MPSETR (IABS(R(SV+2)))
      CALL MPRND (R(I2), TG, X, IABS(R(SV)), 1)
C RESTORE T ETC.
      CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPLNS  ******
      SUBROUTINE MPLNS (X, Y)
C RETURNS MP Y = LN(1+X) IF X IS AN MP NUMBER SATISFYING THE
C CONDITION ABS(X) .LT. 1/B, ERROR OTHERWISE.
C X MAY BE PACKED OR UNPACKED, Y IS UNPACKED.
C USES NEWTONS METHOD TO SOLVE THE EQUATION
C EXP1(-Y) = X, THEN REVERSES SIGN OF Y.
C (HERE EXP1(Y) = EXP(Y) - 1 IS COMPUTED USING MPEXP1).
C TIME IS O(SQRT(T).M(T)) AS FOR MPEXP1.
C ROUNDING IS DEFINED BY THE PARAMETER RNDRL IN COMMON /MPCOM/ -
C RNDRL = 0 OR 1 - ERROR LESS THAN 0.6 UNITS IN LAST PLACE.
C RNDRL = 2 OR 3 - SEE COMMENTS IN SUBROUTINE MPNZR.
      COMMON R
      COMMON /MPCOM/ B, T, DUMMY
      INTEGER IT0, I2, I3, I4, I5, SV, TG, TS2, TS3
      INTEGER B, DUMMY(21), MPGD, MPGET, R(1), T, X(1), Y(1)
C SAVE T ETC. THEN INCREASE, USE TRUNCATION INTERNALLY.
      CALL MPSAVN (SV)
      T = MAX0 (5, T + 1 + MPGD (100))
      TG = T
      CALL MPSETR (0)
C ALLOCATE TEMPORARY SPACE.
      CALL MPNEW (I2)
      CALL MPNEW (I3)
      CALL MPNEW (I4)
      CALL MPNEW (I5)
C CHECK FOR X = 0 EXACTLY
      IF (X(1).NE.0) GO TO 10
      Y(1) = 0
      GO TO 50
C CHECK THAT ABS(X) .LT. 1/B
   10 IF (MPGET (X, 2) .GE. 0) CALL MPERRM (
     $  33HABS(X) .GE. 1/B IN CALL TO MPLNS$)
C GET STARTING APPROXIMATION TO -LN(1+X).
      CALL MPMOVE (X, IABS(R(SV)), R(I3), TG)
      CALL MPSTR (R(I3), R(I5))
      CALL MPDIVI (R(I5), 4, R(I2))
      CALL MPADDQ (R(I2), -1, 3, R(I2))
      CALL MPMUL (R(I5), R(I2), R(I2))
      CALL MPADDQ (R(I2), 1, 2, R(I2))
      CALL MPMUL (R(I5), R(I2), R(I2))
      CALL MPADDI (R(I2), -1, R(I2))
      CALL MPMUL (R(I5), R(I2), R(I5))
C START NEWTON ITERATION USING SMALL T, LATER INCREASE
      T = MAX0 (5, 13 - 2*B)
      IT0 = (T + 5)/2
   20 CALL MPEXP1 (R(I5), R(I4))
      CALL MPMUL (R(I3), R(I4), R(I2))
      CALL MPADD (R(I4), R(I2), R(I4))
      CALL MPADD (R(I3), R(I4), R(I4))
      CALL MPSUB (R(I5), R(I4), R(I5))
      IF (T.GE.TG) GO TO 40
C FOLLOWING LOOP COMPUTES NEXT VALUE OF T TO USE.
C BECAUSE NEWTONS METHOD HAS 2ND ORDER CONVERGENCE,
C WE CAN ALMOST DOUBLE T EACH TIME.
      TS3 = T
      T = TG
   30 TS2 = T
      T = (T+IT0)/2
      IF (T.GT.TS3) GO TO 30
      T = TS2
      GO TO 20
C CHECK THAT NEWTON ITERATION WAS CONVERGING AS EXPECTED
   40 IF ((R(I4).NE.0).AND.((2*R(I4+1)).GT.(IT0-T)))
     $  CALL MPERRM (41HNEWTON ITERATION NOT CONVERGING IN MPLNS$)
C REVERSE SIGN AND ROUND RESULT.
      R(I5) = -R(I5)
      CALL MPSETR (IABS(R(SV+2)))
      CALL MPRND (R(I5), TG, Y, IABS(R(SV)), 1)
C RESTORE STACK POINTER ETC. AND RETURN
   50 CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPL235  ******
      SUBROUTINE MPL235 (I, J, K, X)
C CALLED BY MPLNI, NOT RECOMMENDED FOR INDEPENDENT USE.
C RETURNS MP X = LN((2**I)*(3**J)*(5**K)), FOR INTEGER I, J AND K.
C THE METHOD REQUIRES TIME O(T**2).  LN(81/80), LN(25/24) AND
C LN(16/15) ARE CALCULATED FIRST.  MPL235 COULD BE SPEEDED
C UP IF THESE CONSTANTS WERE PRECOMPUTED AND SAVED.
C ASSUMED THAT I, J AND K NOT TOO LARGE.
C ROUNDING OPTIONS NOT IMPLEMENTED AND NO GUARD DIGITS USED.
      COMMON R
      COMMON /MPCOM/ B, T, DUMMY
      INTEGER C(3), D(3), I2, I3, N, Q, SV, TWO
      INTEGER B, DUMMY(21), I, J, K, MPPARN, R(1), T, X(1)
      DATA D(1), D(2), D(3) /161, 49, 31/
      TWO = 2
C SAVE T ETC. AND ALLOCATE TEMPORARY SPACE
      CALL MPSAVN (SV)
      CALL MPNEW (I2)
      CALL MPNEW (I3)
C USE TRUNCATION INTERNALLY.
      CALL MPSETR (0)
      X(1) = 0
      IF (MAX0(IABS(I), IABS(J), IABS(K)) .GE. (MPPARN(16)/68))
     $  CALL MPERRM (37HI, J OR K TOO LARGE IN CALL TO MPLNI$)
      C(1) = 3*I + 5*J + 7*K
      C(2) = 5*I + 8*J + 12*K
      C(3) = 7*I + 11*J + 16*K
      DO 20 Q = 1, 3
      T = R(SV)
      CALL MPCQM (2*C(Q), D(Q), R(I2))
      CALL MPSTR (R(I2), R(I3))
      CALL MPADD (X, R(I3), X)
      N = 1
   10 N = N + 2
      IF (R(I2).EQ.0) GO TO 20
C REDUCE T IF POSSIBLE
      T = R(SV) + R(I2+1) + 2 - X(TWO)
      IF (T.LE.2) GO TO 20
      T = MIN0 (T, R(SV))
      CALL MPMULS (R(I2), 1, 1, D(Q), D(Q))
      CALL MPDIVI (R(I2), N, R(I3))
      T = R(SV)
      CALL MPADD (R(I3), X, X)
      GO TO 10
   20 CONTINUE
C RESTORE STACK POINTER ETC. AND RETURN
      CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPMAXR  ******
      SUBROUTINE MPMAXR (X)
C SETS X TO THE LARGEST POSSIBLE POSITIVE MP NUMBER
      COMMON /MPCOM/ B, T, M, DUMMY
      INTEGER B, DUMMY(20), I, IT, M, T, TWO, X(1)
      TWO = 2
C CHECK LEGALITY OF PARAMETERS IN COMMON /MPCOM/
      CALL MPCHK
      IT = B - 1
C SET FRACTION DIGITS TO B-1
      DO 10 I = 1, T
   10 X(I+2) = IT
C SET SIGN AND EXPONENT, AND UPDATE MXEXPN.
      X(1) = 1
      X(TWO) = M
      CALL MPUPDT (X(TWO))
      RETURN
      END
C $$                   ******  MPMINR  ******
      SUBROUTINE MPMINR (X)
C SETS X TO THE SMALLEST POSITIVE NORMALIZED MP NUMBER B**(-M)
      COMMON /MPCOM/ B, T, M, DUMMY
      INTEGER B, DUMMY(20), I, IL, M, T, TWO, X(1)
      TWO = 2
C CLEAR X
      IL = T + 2
      DO 10 I = 1, IL
   10 X(I) = 0
C SET SIGN, EXPONENT AND FIRST DIGIT
      X(1) = 1
      X(TWO) = 1 - M
      X(TWO+1) = 1
C UPDATE MNEXPN AND RETURN
      CALL MPUPDT (X(TWO))
      RETURN
      END
C $$                   ******  MPMLP  ******
      SUBROUTINE MPMLP (U, V, W, J)
C PERFORMS INNER MULTIPLICATION LOOP FOR MPMUL
C NOTE THAT CARRIES ARE NOT PROPAGATED IN INNER LOOP,
C WHICH SAVES TIME AT THE EXPENSE OF SPACE.
      INTEGER I, J, U(1), V(1), W
      DO 10 I = 1, J
   10 U(I) = U(I) + W*V(I)
      RETURN
      END
C $$                   ******  MPMOVE  ******
      SUBROUTINE MPMOVE (X, TX, Y, TY)
C ASSUMES MP NUMBER X OF TX .GE. 2 DIGITS,
C MP NUMBER Y OF TY .GE. 2 DIGITS (Y NOT NECESSARILY INITIALIZED).
C MOVES X TO Y, EITHER PADDING WITH TRAILING ZEROS OR ROUNDING
C (AS SPECIFIED BY RNDRL - SEE SUBROUTINE MPNZR) DEPENDING ON
C WHETHER TX IS LESS OR GREATER THAN TY.  THE VALUE OF T IN
C COMMON /MPCOM/ IS RESTORED ON RETURN.
C X MAY BE IN PACKED OR UNPACKED FORMAT, Y IS UNPACKED.
      COMMON R
      COMMON /MPCOM/ B, T, DUMMY
      INTEGER EX, I, IH, IL, IY, I2, KG, SN, SV
      INTEGER B, DUMMY(21), R(1), T, TX, TY, X(1), Y(1)
      CALL MPSAVN (SV)
C USE LOCAL VARIABLES IN CASE T ACTUAL ARGUMENT
      KG = TX - TY
      IY = TY
      T = TX
C COMPARE TX AND TY.
      IF (KG.GT.0) GO TO 20
C HERE TX .LE. TY
      CALL MPUNPK (X, Y)
      IF (KG.EQ.0) GO TO 40
C HERE TX .LT. TY.
      IL = T + 3
      IH = IY + 2
C PAD WITH TRAILING ZEROS.
      DO 10 I = IL, IH
   10 Y(I) = 0
      GO TO 40
C HERE TX. GT. TY.   CHECK FOR X ZERO.
   20 IF (X(1).NE.0) GO TO 30
C HERE X IS ZERO SO SET Y TO ZERO AND RETURN
      Y(1) = 0
      GO TO 40
C MOVE X TO TEMPORARY STORAGE AND ROUND
   30 CALL MPNEW (I2)
      CALL MPUNPK (X, R(I2))
C REDUCE T FOR CALL TO MPNZR.
      T = IY
      SN = R(I2)
      EX = R(I2+1)
      CALL MPNZR (SN, EX, Y, R(I2+2), KG)
C RESTORE STACK POINTER ETC.
   40 CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPMUL  ******
      SUBROUTINE MPMUL (X, Y, Z)
C
C MULTIPLIES X AND Y, RETURNING RESULT IN Z, FOR MP X, Y AND Z.
C THE SIMPLE O(T**2) ALGORITHM IS USED, WITH ROUNDING DEFINED BY THE
C PARAMETER RNDRL IN COMMON /MPCOM/, AS FOLLOWS -
C RNDRL = 0 - TRUNCATE RESULT TOWARDS ZERO,
C             ERROR LESS THAN 1.01 UNITS IN THE LAST PLACE.
C RNDRL = 1, 2 OR 3 - SEE SUBROUTINE MPNZR.
C
C ADVANTAGE IS TAKEN OF ZERO DIGITS IN X, BUT NOT IN Y.
C ASYMPTOTICALLY FASTER ALGORITHMS ARE KNOWN (SEE KNUTH,
C VOL. 2), BUT ARE DIFFICULT TO IMPLEMENT IN FORTRAN IN AN
C EFFICIENT AND MACHINE-INDEPENDENT MANNER.
C IN COMMENTS TO OTHER MP ROUTINES, M(T) IS THE TIME
C TO PERFORM T-DIGIT MP MULTIPLICATION.   THUS
C M(T) = O(T**2) WITH THE PRESENT VERSION OF MPMUL,
C BUT M(T) = O(T.LOG(T).LOG(LOG(T))) IS THEORETICALLY POSSIBLE.
C
C MP OVER/UNDERFLOW IS DETECTED BY SUBROUTINE MPNZR.
C
      COMMON R
      COMMON /MPCOM/ B, T, DUMMY
      INTEGER C, I, IH, IP, I2, J, J1, KG, RE, RI, RS, TG, TWO,
     $  B, DUMMY(21), NEXT, R(1), SV, T, XI, X(1), Y(1), Z(1)
      TWO = 2
C FORM SIGN OF PRODUCT
      RS = X(1)*Y(1)
      IF (RS.NE.0) GO TO 10
C SET RESULT TO ZERO
      Z(1) = 0
      RETURN
C COMPUTE NUMBER OF GUARD DIGITS REQUIRED
   10 CALL MPSAVN (SV)
      KG = MIN0 (3, T)
      IF ((R(SV+2) .NE. 0) .OR. (T .GT. ((B*B)/100))) KG = T
C ALLOCATE SPACE FOR ACCUMULATOR, CHECK PARAMETERS ETC.
      TG = T + KG
      CALL MPNEW2 (I2, TG)
      NEXT = I2 + TG
      IH = NEXT - 1
C FORM EXPONENT OF PRODUCT
      RE = X(TWO) + Y(TWO)
C CLEAR ACCUMULATOR
      DO 20 I = I2, IH
   20 R(I) = 0
C PERFORM MULTIPLICATION
      C = 8
      DO 40 I = 1, T
      XI = X(I+2)
C FOR SPEED, PUT THE NUMBER WITH MANY ZEROS FIRST
      IF (XI.EQ.0) GO TO 40
      IP = I + I2
      CALL MPMLP (R(IP), Y(TWO+1), XI, MIN0 (T, TG - I))
      C = C - 1
      IF (C.GT.0) GO TO 40
C CHECK FOR LEGAL BASE B DIGIT
      IF ((XI.LT.0).OR.(XI.GE.B)) GO TO 80
C PROPAGATE CARRIES AT END AND EVERY EIGHTH TIME,
C FASTER THAN DOING IT EVERY TIME.
      DO 30 J = 1, TG
      J1 = NEXT - J
      RI = R(J1) + C
      IF (RI.LT.0) GO TO 70
      C = RI/B
   30 R(J1) = RI - B*C
      IF (C.NE.0) GO TO 80
      C = 8
   40 CONTINUE
      IF (C.EQ.8) GO TO 60
      IF ((XI.LT.0).OR.(XI.GE.B)) GO TO 80
      C = 0
      DO 50 J = 1, TG
      J1 = NEXT - J
      RI = R(J1) + C
      IF (RI.LT.0) GO TO 70
      C = RI/B
   50 R(J1) = RI - B*C
      IF (C.NE.0) GO TO 80
C NORMALIZE AND ROUND RESULT
   60 CALL MPNZR (RS, RE, Z, R(I2), KG)
C RESTORE STACK POINTER AND RETURN
      CALL MPRESN (SV)
      RETURN
   70 CALL MPERRM (39HINTEGER OVERFLOW IN MPMUL, B TOO LARGE$)
   80 CALL MPERRM (38HILLEGAL BASE B DIGIT IN CALL TO MPMUL$)
      RETURN
      END
C $$                   ******  MPMULI  ******
      SUBROUTINE MPMULI (X, IY, Z)
C MULTIPLIES MP X BY SINGLE-PRECISION INTEGER IY GIVING MP Z.
C ROUNDING DEFINED BY PARAMETER RNDRL IN COMMON /MPCOM/ -
C SEE SUBROUTINE MPNZR.
C MULTIPLICATION BY 1 MAY BE USED TO NORMALIZE A NUMBER
C EVEN IF LAST DIGIT IS GREATER THAN B-1.
      COMMON R
      COMMON /MPCOM/ B, T, M, LUN, MXR, SPTR, DUMMY
      INTEGER C, C1, C2, I, IJ, IKG, IP, IS, IX, I2, J, J1, J2, KG, RE
      INTEGER RI, RS, TG, TWO, T1
      INTEGER B, DUMMY(17), IY, LUN, M, MXR, R(1), SPTR, T, X(1), Z(1)
      INTEGER MPGD, MPPARN
      TWO = 2
C COMPUTE NUMBER OF GUARD DIGITS REQUIRED TO GIVE EXACT PRODUCT.
      KG = MPGD (IY)
C IF IY IS 1 LAST DIGIT MAY BE LARGE (SEE E.G. MPCIM).
      IF (IY.EQ.1) KG = 1
C ALLOCATE T + KG WORDS FOR ACCUMULATOR
      TG = T + KG
      CALL MPNEW2 (I2, TG)
C CHECK FOR X ZERO
      RS = X(1)
      IF (RS.EQ.0) GO TO 110
C CHECK FOR MULTIPLIER ZERO.
      J = IY
      IF (J) 10, 110, 20
C HERE ADJUST SIGN FOR NEGATIVE MULTIPLIER.
   10 J = -J
      RS = -RS
C CHECK FOR MULTIPLICATION BY +-B
   20 IF (J.NE.B) GO TO 30
C CHECK FOR OVERFLOW.
      IF (X(TWO) .GE. M) CALL MPOVFL (Z)
C HERE MULTIPLICATION BY +-B, RESULT DOES NOT OVERFLOW.
      CALL MPSTR (X, Z)
      Z(1) = RS
      Z(TWO) = Z(TWO) + 1
      CALL MPUPDT (Z(TWO))
      GO TO 120
C HAVE TO DO GENUINE MULTIPLICATION HERE.
C SET EXPONENT TO EXPONENT(X) + NUMBER OF GUARD DIGITS.
   30 RE = X(TWO) + KG
C FORM PRODUCT IN ACCUMULATOR
      C = 0
      T1 = T + 1
C IF J*B NOT REPRESENTABLE AS AN INTEGER WE HAVE TO SIMULATE
C DOUBLE-PRECISION MULTIPLICATION.
      IF (J.GT.(MPPARN(16)/B)) GO TO 80
C INNER LOOP FOR SMALL ABS(IY) CASE.
      DO 40 IJ = 1, T
      I = T1 - IJ
      RI = J*X(I+2) + C
      C = RI/B
      IP = SPTR - IJ
   40 R(IP) = RI - B*C
C CHECK FOR INTEGER OVERFLOW
      IF (RI.LT.0) GO TO 100
C HAVE TO TREAT FIRST KG WORDS OF ACCUMULATOR SEPARATELY
      IKG = I2 + KG
      IF (KG.LE.0) GO TO 60
      DO 50 IJ = 1, KG
      I = IKG - IJ
      RI = C
      C = RI/B
   50 R(I) = RI - B*C
   60 IF (C.NE.0) GO TO 100
C NORMALIZE AND ROUND OR TRUNCATE RESULT
   70 CALL MPNZR (RS, RE, Z, R(I2), KG)
      GO TO 120
C HERE J MAY BE TOO LARGE FOR SINGLE-PRECISION MULTIPLICATION
   80 J1 = J/B
      J2 = J - J1*B
C FORM PRODUCT
      DO 90 IJ = 1, TG
      C1 = C/B
      C2 = C - B*C1
      I = T1 - IJ
      IX = 0
      IF (I.GT.0) IX = X(I+2)
      RI = J2*IX + C2
      IS = RI/B
      C = J1*IX + C1 + IS
      IP = SPTR - IJ
   90 R(IP) = RI - B*IS
      IF (C.EQ.0) GO TO 70
C CAN ONLY GET HERE IF INTEGER OVERFLOW OCCURRED OR B**T TOO SMALL.
  100 CALL MPERRM (25HERROR OCCURRED IN MPMULI$)
C SET RESULT TO ZERO
  110 Z(1) = 0
C RESTORE STACK POINTER
  120 SPTR = I2
      RETURN
      END
C $$                   ******  MPMULQ  ******
      SUBROUTINE MPMULQ (X, I, J, Y)
C MULTIPLIES MP X BY I/J, GIVING MP RESULT Y.
C ROUNDING DEFINED BY PARAMETER RNDRL IN COMMON /MPCOM/ AS FOLLOWS -
C RNDRL = 0 - AS FOR MPMULI FOLLOWED BY MPDIVI WITH RNDRL = 0.
C             SEE COMMENTS IN MPDIVI, MPMULI AND MPNZR.  ERROR IS
C             LESS THAN B UNITS IN THE LAST PLACE.
C RNDRL = 1, 2, OR 3 - SEE COMMENTS IN SUBROUTINE MPNZR.
C OVERFLOW/UNDERFLOW MAY OCCUR IN MPMULI/MPDIVI RESPECTIVELY.
      COMMON R
      COMMON /MPCOM/ B, T, M, LUN, MXR, SPTR, MXSPTR, MNSPTR,
     $  MXEXPN, MNEXPN, RNDRL, DUMMY
      INTEGER IS, I2, JS, MPGD, TG, TS
      INTEGER B, DUMMY(12), I, J, LUN, M, MNEXPN, MNSPTR, MXEXPN, MXR
      INTEGER MXSPTR, R(1), RNDRL, SPTR, T, X(1), Y(1)
C CHECK FOR DIVISION BY ZERO.
      IF (J .EQ. 0) CALL MPERRM (
     $  37HATTEMPTED DIVISION BY ZERO IN MPMULQ$)
C CHECK FOR MULTIPLICATION BY ZERO.
      IF (I.NE.0) GO TO 10
      Y(1) = 0
      RETURN
C REDUCE TO LOWEST TERMS
   10 IS = I
      JS = J
      CALL MPGCD (IS, JS)
      IF (IABS(IS).EQ.1) GO TO 30
C CHECK ROUNDING OPTIONS
      IF (RNDRL.GT.0) GO TO 20
C HERE RNDRL = 0, SO USE SIMPLE METHOD.
      CALL MPMULI (X, IS, Y)
      CALL MPDIVI (Y, JS, Y)
      RETURN
C HERE RNDRL. GT. 0, SO USE ENOUGH GUARD DIGITS THAT MULTIPLICATION
C IS EXACT.
   20 TS = T
      T = T + MPGD (IS)
      TG = T
      CALL MPNEW (I2)
      CALL MPMOVE (X, TS, R(I2), TG)
      CALL MPMULI (R(I2), IS, R(I2))
      CALL MPDIVI (R(I2), JS, R(I2))
C ROUND RESULT
      CALL MPMOVE (R(I2), TG, Y, TS)
C RESTORE T AND STACK POINTER
      T = TS
      SPTR = I2
      RETURN
C HERE IS = +-1
   30 CALL MPDIVI (X, IS*JS, Y)
      RETURN
      END
C $$                   ******  MPMULS  ******
      SUBROUTINE MPMULS (X, I, J, K, L)
C SETS X = X*I*J/(K*L) FOR MP X, INTEGER I, J, K, L,
C K*L NONZERO.  CALLS MPMULQ ONCE IF I*J AND K*L NOT TOO LARGE,
C OTHERWISE MPMULQ CALLED TWICE.
C ROUNDING NOT BEST POSSIBLE, BUT DIRECTED ROUNDING
C IS IN THE CORRECT DIRECTION.
      INTEGER IJ, IS, KL, KS, MPPARN, MXINT
      INTEGER I, J, K, L, X(1)
      IF ((J.EQ.0).OR.(L.EQ.0)) GO TO 30
C SEE IF I*J OR K*L MIGHT OVERFLOW.
      MXINT = MPPARN(16)
      IF ((IABS(I) .GT. (MXINT/IABS(J))) .OR.
     $    (IABS(K) .GT. (MXINT/IABS(L)))) GO TO 30
C HERE SAFE TO FORM I*J AND K*L.
      IJ = I*J
      KL = K*L
C WORTHWHILE TO CHECK FOR SPECIAL CASES HERE.
      IF (IJ.NE.1) GO TO 10
      CALL MPDIVI (X, KL, X)
      RETURN
   10 IF (KL.NE.1) GO TO 20
      CALL MPMULI (X, IJ, X)
      RETURN
   20 CALL MPMULQ (X, IJ, KL, X)
      RETURN
C HERE SPLIT UP INTO TWO CALLS, TAKING CARE FOR DIRECTED ROUNDING.
   30 IS = I
      IF (J.LT.0) IS = -IS
      KS = K
      IF (L.LT.0) KS = -KS
      CALL MPMULQ (X, IS, KS, X)
      CALL MPMULQ (X, IABS(J), IABS(L), X)
      RETURN
      END
C $$                   ******  MPNEW  ******
      SUBROUTINE MPNEW (I)
C RETURNS INDEX I SUCH THAT WORDS I, ... , I+T+1 OF BLANK COMMON
C ARE AVAILABLE FOR USE, UPDATES STACK POINTER SPTR ETC.
      COMMON /MPCOM/ B, T, DUMMY
      INTEGER B, DUMMY(21), I, T
      CALL MPNEW2 (I, T+2)
      RETURN
      END
C $$                   ******  MPNEW2  ******
      SUBROUTINE MPNEW2 (I, J)
C RETURNS I SUCH THAT WORDS I, ... , I+ABS(J)-1 OF BLANK COMMON
C ARE AVAILABLE FOR USE, UPDATES STACK POINTER SPTR ETC.
      COMMON /MPCOM/ B, T, M, LUN, MXR, SPTR, MXSPTR, MNSPTR, DUMMY
      INTEGER B, DUMMY(15), I, J, LUN, M, MXR, MXSPTR, MNSPTR, SPTR, T
C ALLOCATE SPACE ON STACK
      I = SPTR
      SPTR = I + IABS(J)
C CHECK FOR STACK OVERFLOW AND UPDATE MXSPTR.
C USEFUL TO CHECK VALIDITY OF PARAMETERS IN COMMON /MPCOM/
C IF TOP LEVEL CALL, TOO.
      IF ((SPTR.GT.MXSPTR) .OR. (MNSPTR.GE.I)) CALL MPCHK
      RETURN
      END
C $$                   ******  MPNZR  ******
      SUBROUTINE MPNZR (RS, EXPN, Z, A, KG)
C
C ASSUMES (T+KG)-DIGIT FRACTION IN A(1), ... , A(T+KG),
C KG .GE. 0, SIGN = RS, EXPONENT = EXPN.
C A(1) AND Z(3) MAY HAVE THE SAME ADDRESS.
C NORMALIZES AND RETURNS MP RESULT IN Z.  A IS DESTROYED.
C
C ROUNDING DEPENDS ON VALUE OF RNDRL (IN COMMON /MPCOM/) AS FOLLOWS -
C
C   0 - ROUND TOWARDS ZERO (I.E. CHOP OR TRUNCATE).  THE RESULTING
C       ERROR IS LESS THAN 1 ULP (UNIT IN THE LAST PLACE).
C       THIS IS THE DEFAULT (IF MPSET USED).  IT IS THE FASTEST OPTION
C       AND IS SATISFACTORY FOR MOST PURPOSES.
C       NOTE - RNDRL = 0 IS ASSUMED IF KG IS ZERO.
C
C   1 - ROUND TO NEAREST, PREFERRING EVEN LAST DIGIT IF TIE.
C       MUST HAVE KG.GT.0 TO BE MEANINGFUL.  ERROR .LE. 1/2 ULP IF
C       KG (THE NUMBER OF GUARD DIGITS) IS LARGE ENOUGH.
C       THIS OPTION GIVES WORST-CASE ERROR BOUNDS HALF AS LARGE AS
C       FOR RNDRL = 0, AND AVERAGE-CASE ERROR CONSIDERABLY BETTER
C       THAN FOR RNDRL = 0.  HOWEVER, IF RNDRL = 0 DOES NOT GIVE
C       SUFFICIENT ACCURACY IS IS USUALLY PREFERABLE TO INCREASE T
C       RATHER THAN USE RNDRL = 1.
C
C   2 - ROUND DOWN (TOWARDS -INFINITY).  (MUST HAVE KG .GT. 0)
C
C   3 - ROUND UP   (TOWARDS +INFINITY).  (MUST HAVE KG .GT. 0)
C       THESE LAST TWO OPTIONS ARE USEFUL FOR INTERVAL ARITHMETIC AS
C       THEY GIVE RIGOROUS LOWER AND UPPER BOUNDS.
C
C NOTE - COMMENTS IN OTHER ROUTINES OFTEN REFER TO THE ABOVE
C        DEFINITIONS OF ROUNDING OPTIONS.  THE COMMENTS REGARDING
C        KG ABOVE REFER ONLY TO THIS SUBROUTINE.
C
      COMMON /MPCOM/ B, T, M, LUN, MXR, SPTR, MXSPTR, MNSPTR,
     $  MXEXPN, MNEXPN, RNDRL, DUMMY
      INTEGER B2, I, INM, INP, IS, J, K, RE, TG, TWO, T1
      INTEGER A(1), B, DUMMY(12), EXPN, KG, LUN, M, MNEXPN, MNSPTR
      INTEGER MXEXPN, MXR, MXSPTR, RNDRL, RS, SPTR, T, Z(1)
      TWO = 2
C CHECK THAT SIGN = +-1
C THE FOLLOWING MESSAGE IS USUALLY CAUSED BY EXCEEDING
C ARRAY BOUNDS.  CHECK THAT ARRAYS USED FOR MP NUMBERS HAVE
C DIMENSION AT LEAST T+2.
      IF (IABS(RS) .GT. 1) CALL MPERRM (
     $  38HSIGN NOT 0, +1 OR -1 IN CALL TO MPNZR$)
C CHECK FOR ZERO
      IF (RS .NE. 0) GO TO 20
C RETURN ZERO.
   10 Z(1) = 0
      RETURN
C SAVE EXPONENT LOCALLY
   20 RE = EXPN
      T1 = T + 1
      TG = T + KG
C LOOK FOR FIRST NONZERO DIGIT
      DO 30 I = 1, TG
      IS = I - 1
      IF (A(I).GT.0) GO TO 40
   30 CONTINUE
C FRACTION ZERO
      GO TO 10
C SEE IF NUMBER IS NORMALIZED.
   40 IF (IS.EQ.0) GO TO 70
C NORMALIZE BY LEFT SHIFTING AND PADDING ON RIGHT WITH ZEROS.
      RE = RE - IS
      INM = TG - IS
      DO 50 J = 1, INM
      K = J + IS
   50 A(J) = A(K)
      INP = INM + 1
      DO 60 J = INP, TG
   60 A(J) = 0
C CHECK VARIOUS ROUNDING OPTIONS DESCRIBED ABOVE.
   70 IF ((RNDRL.EQ.0).OR.(KG.EQ.0).OR.
     $   ((RNDRL.EQ.2).AND.(RS.GT.0)).OR.
     $   ((RNDRL.EQ.3).AND.(RS.LT.0))) GO TO 160
      IF (RNDRL.EQ.1) GO TO 90
C HERE ROUND AWAY FROM ZERO IF ANY OF A(T+1), ... , A(T+KG) NONZERO.
      DO 80 I = T1, TG
      IF (A(I).NE.0) GO TO 120
   80 CONTINUE
C HERE A(T+1) = ... = A(T+KG) = 0 SO CAN TRUNCATE.
      GO TO 160
C ROUND TO NEAREST HERE.  NEED TO TREAT ODD AND EVEN BASES DIFFERENTLY.
   90 B2 = B/2
      IF ((2*B2).NE.B) GO TO 140
C B EVEN.  ROUND AWAY FROM ZERO IF A(T+1).GE.B2 UNLESS
C A(T) EVEN, A(T+1) = B2, AND A(T+2) = ... = A(T+KG) = 0.
      IF (A(T1) - B2) 160, 100, 120
  100 IF (MOD(A(T),2).NE.0) GO TO 120
      I = T1
  110 I = I + 1
      IF (I.GT.TG) GO TO 160
      IF (A(I).EQ.0) GO TO 110
C ROUND AWAY FROM ZERO
  120 DO 130 J = 1, T
      I = T1 - J
      A(I) = A(I) + 1
      IF (A(I).LT.B) GO TO 160
  130 A(I) = 0
C EXCEPTIONAL CASE, ROUNDED UP TO .10000...
      RE = RE + 1
      A(1) = 1
      GO TO 160
C ODD BASE, ROUND AWAY FROM ZERO IF .A(T+1) ... A(T+KG) .GT. 1/2
  140 DO 150 I = T1, TG
      IF (A(I) - B2) 160, 150, 120
  150 CONTINUE
C CHECK FOR OVERFLOW
  160 IF (RE .GT. M) CALL MPOVFL (Z)
C CHECK FOR UNDERFLOW
      IF (RE.LE.(-M)) GO TO 180
C STORE RESULT IN Z
      Z(1) = RS
      Z(TWO) = RE
      DO 170 I = 1, T
  170 Z(I+2) = A(I)
C UPDATE MAXIMUM AND MINIMUM EXPONENT INDICATORS
      MXEXPN = MAX0 (MXEXPN, RE)
      MNEXPN = MIN0 (MNEXPN, RE)
      RETURN
C UNDERFLOW HERE
  180 CALL MPUNFL (Z)
      RETURN
      END
C $$                   ******  MPOUT  ******
      SUBROUTINE MPOUT (X, C, P, N)
C
C CONVERTS MULTIPLE-PRECISION X TO FP.N FORMAT IN C,
C WHICH MAY BE PRINTED UNDER PA1 FORMAT.  NOTE THAT
C N = -1 IS ALLOWED, AND EFFECTIVELY GIVES IP FORMAT.
C DIGITS AFTER THE DECIMAL POINT ARE BLANKED OUT IF
C THEY COULD NOT BE SIGNIFICANT.
C EFFICIENCY IS HIGHER IF B IS A POWER OF 10 THAN IF NOT.
C X MAY BE PACKED OR UNPACKED.
C DIMENSION OF C MUST BE AT LEAST P.
C C IS AN INTEGER ARRAY, P AND N ARE INTEGERS.
C
C ROUNDING OF DECIMAL OUTPUT IS DETERMINED BY THE PARAMETER
C RNDRL IN COMMON /MPCOM/, AS FOLLOWS -
C
C CALL MPSETR (0) OR 1 - ROUND TO APPROXIMATLELY NEAREST DECIMAL NUMBER
C                  REPRESENTABLE IN THE SPECIFIED FORMAT.
C                  ERROR IN CONVERSION TO DECIMAL IS LESS
C                  THAN 0.6 UNITS IN THE LAST DECIMAL PLACE.
C RNDRL = 2 -      ROUND DOWN (TOWARDS -INFINITY) TO A DECIMAL
C                  NUMBER REPRESENTABLE IN THE SPECIFIED FORMAT.
C RNDRL = 3 -      ROUND UP (TOWARDS +INFINITY) TO A DECIMAL NUMBER
C                  REPRESENTABLE IN THE SPECIFIED FORMAT.
C
C OUTPUT BASE IS USUALLY 10 (DEFAULT SET BY MPSET), BUT
C MAY BE CHANGED TO ANY OF 2, ... , 16 BY CHANGING THE
C PARAMETER OUTBAS IN COMMON /MPCOM/ (SEE MPPARN).
C
      COMMON R
      COMMON /MPCOM/ B, T, DUMMY
      INTEGER I, IB, IP, IR, IS, ISZ, ITP, IZ, I1, I2, I3, J, JD,
     $  JP, NMAX, NP, OUTBAS, SV, TG, TP,
     $  B, C(1), DUMMY(21), N, P, R(1), T, X(1),
     $  MPCHGB, MPDIGW, MPGD, MPPARN,
     $  BLANK, PERIOD, MINUS, STAR
      DATA BLANK, PERIOD, MINUS, STAR /1H , 1H., 1H-, 1H*/
C CHECK B, T, OUTBAS ETC. FOR LEGALITY.
      CALL MPCHK
C SAVE T ETC.
      CALL MPSAVN (SV)
C GET OUTPUT BASE.
      OUTBAS = MPPARN (20)
C CHECK LEGALITY OF PARAMETERS N AND P.
      IF ((N .LT. (-1)) .OR. (P .LE. N) .OR. (P .LE. 0))
     $  CALL MPERRM (36HN AND/OR P ILLEGAL IN CALL TO MPOUT$)
C COMPUTE DISPLACEMENTS.
      NP = P - N
      IP = NP - 1
C COMPUTE POWER OF OUTBAS WHICH WE CAN SAFELY MULTIPLY AND
C DIVIDE BY (THIS SAVES TIME).
      IB = MPPARN(16)/OUTBAS
      TP = OUTBAS
      ITP = 1
   10 IF ((TP .GT. IB).OR.(TP .EQ. B)) GO TO 20
      TP = TP*OUTBAS
      ITP = ITP + 1
      GO TO 10
C COMPUTE GUARD DIGITS REQUIRED, MOVE X ETC.
   20 T = T + 1 + MPGD (MAX0 (50, TP))
      TG = T
      CALL MPNEW (I2)
      CALL MPNEW (I3)
      CALL MPMOVE (X, IABS(R(SV)), R(I2), TG)
C USE TRUNCATION (USUALLY) FOR INTERNAL ARITHMETIC
      CALL MPSETR (0)
C PUT FORMATTED ZERO IN C
      DO 30 I = 1, P
      C(I) = BLANK
      IF (I .GE. IP) C(I) = MPDIGW(0)
      IF (I .EQ. NP) C(I) = PERIOD
   30 CONTINUE
C GET SIGN OF X, CHECK FOR ZERO
      IS = R(I2)
      IF (IS .EQ. 0) GO TO 160
      R(I2) = 1
C COMPUTE MAXIMUM NUMBER OF NONZERO DIGITS WHICH WE CAN
C MEANINGFULLY GIVE AFTER DECIMAL POINT.
      NMAX = MAX0 (0, MIN0 (N, MPCHGB (OUTBAS, IABS(B), MIN0 (4*N,
     $  MAX0 (0, R(SV) - R(I2+1))))))
C COMPUTE ROUNDING CONSTANT FOR DECIMAL OUTPUT
      IR = 1
      IF (R(SV+2) .GE. 2) IR = 2*(R(SV+2)-2)
      IF (IS.LT.0) IR = 2 - IR
C CHANGE ROUNDING OPTION IF NECESSARY FOR DIRECTED ROUNDING
      IF (IR .EQ. 2) CALL MPSETR (3)
      CALL MPCQM (IR, 2, R(I3))
      IF (NMAX .GT. 0) CALL MPSCAL (R(I3), OUTBAS, (-NMAX))
C ADD ROUNDING CONSTANT TO ABS(X).
      CALL MPADD (R(I2), R(I3), R(I2))
C IP PLACES BEFORE POINT, SO DIVIDE BY OUTBAS**IP
      IF (IP .GT. 0) CALL MPSCAL (R(I2), OUTBAS, (-IP))
      IZ = 0
C ROUND RESULT TO R(SV)+1 DIGITS FOR DIRECTED ROUNDINGS (SO SUBSEQUENT
C ARITHMETIC EXACT).
      IF (R(SV+2) .LE. 1) GO TO 40
      CALL MPMOVE (R(I2), TG, R(I2), R(SV)+1)
      CALL MPMOVE (R(I2), R(SV)+1, R(I2), TG)
C NOW USE CHOPPED ARITHMETIC
   40 CALL MPSETR (0)
C CHECK THAT NUMBER IS LESS THAN ONE
      IF (R(I2+1) .GT. 0) GO TO 120
      IF (IP .LE. 0) GO TO 90
C PUT DIGITS BEFORE POINT IN
      JD = 1
      DO 80 I = 1, IP
      IF (JD .GT. 1) GO TO 70
      IF ((I+ITP) .LE. (IP+1)) GO TO 50
C MULTIPLY BY OUTBAS, TRUNCATING RESULT
      CALL MPMULI (R(I2), OUTBAS, R(I2))
      JD = OUTBAS
      GO TO 60
C HERE WE CAN MULTIPLY BY A POWER OF OUTBAS TO SAVE TIME
   50 CALL MPMULI (R(I2), TP, R(I2))
      JD = TP
C GET INTEGER PART
   60 CALL MPCMI (R(I2), JP)
C AND FRACTIONAL PART
      CALL MPCMF (R(I2), R(I2))
   70 JD = JD/OUTBAS
C GET NEXT DECIMAL DIGIT
      J = JP/JD
      JP = JP - J*JD
      ISZ = IZ
      IF ((J .GT. 0).OR.(I .EQ. IP)) IZ = 1
      IF (IZ .GT. 0) C(I) = MPDIGW(J)
      IF ((IZ .EQ. ISZ).OR.(IS .GT. 0)) GO TO 80
      IF (I .EQ. 1) GO TO 120
      C(I-1) = MINUS
   80 CONTINUE
   90 IF (NMAX .LE. 0) GO TO 140
C PUT IN DIGITS AFTER DECIMAL POINT
      JD = 1
      DO 110 I = 1, NMAX
      IF (JD .GT. 1) GO TO 100
      CALL MPMULI (R(I2), TP, R(I2))
      CALL MPCMI (R(I2), JP)
      CALL MPCMF (R(I2), R(I2))
      JD = TP
  100 JD = JD/OUTBAS
      J = JP/JD
      JP = JP - J*JD
      I1 = NP + I
  110 C(I1) = MPDIGW(J)
      GO TO 140
C ERROR OCCURRED, RETURN ASTERISKS.
  120 DO 130 I = 1, P
  130 C(I) = STAR
      GO TO 160
C BLANK OUT ANY NONSIGNIFICANT TRAILING ZEROS
  140 IF (NMAX .GE. N) GO TO 160
      I1 = NP + NMAX + 1
      DO 150 I = I1, P
  150 C(I) = BLANK
C RESTORE STACK POINTER ETC. AND RETURN.
  160 CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPOVFL  ******
      SUBROUTINE MPOVFL (X)
C CALLED ON MULTIPLE-PRECISION OVERFLOW, I.E. WHEN THE
C EXPONENT OF MP NUMBER X WOULD EXCEED M.
C EXECUTION IS TERMINATED WITH AN ERROR MESSAGE
C AFTER CALLING MPMAXR(X) AND SETTING MXEXPN = M + 1.
      COMMON /MPCOM/ B, T, M, LUN, MXR, SPTR, MXSPTR, MNSPTR,
     $  MXEXPN, DUMMY
      INTEGER B, DUMMY(14), LUN, M, MNSPTR, MXEXPN, MXR, MXSPTR, SPTR,
     $  T, X(1)
C M MAY HAVE BEEN OVERWRITTEN, SO CHECK B, T, M ETC.
      CALL MPCHK
C SET X TO LARGEST POSSIBLE POSITIVE NUMBER
      CALL MPMAXR (X)
C SET MXEXPN TO M + 1 (LARGEST VALID EXPONENT IS M)
      MXEXPN = M + 1
      CALL MPERRM (37HCALL TO MPOVFL, MP OVERFLOW OCCURRED$)
C MPERRM DOES NOT RETURN, SO NEVER GET HERE.
      RETURN
      END
C $$                   ******  MPPARC  ******
      SUBROUTINE MPPARC (N, J)
C SETS THE N-TH WORD OF COMMON /MPCOM/ TO J IF 1 .LE. N .LE. 23,
C ERROR OTHERWISE.  FASTER BUT LESS MNEMONIC THAN MPPARB.
C FOR THE MEANINGS OF WORDS 1 TO 23 OF COMMON /MPCOM/, SEE
C THE COMMENTS IN SUBROUTINE MPPARM.  FOR THEIR DEFAULT
C SETTINGS SEE MPSET.
      COMMON /MPCOM/ MPC
      INTEGER J, MPC(23), N
      IF ((N .LE. 0) .OR. (N .GT. 23)) CALL MPERRM (
     $  28HILLEGAL N IN CALL TO MPPARC$)
      MPC(N) = J
C CHECK LEGALITY OF T ETC. IF CALLED FROM TOP LEVEL,
C I.E. IF SPTR .LE. MNSPTR.
      IF (MPC(6) .LE. MPC(8)) CALL MPCHK
      RETURN
      END
C $$                   ******  MPPARN  ******
      INTEGER FUNCTION MPPARN (N)
C RETURNS THE N-TH WORD OF COMMON /MPCOM/ IF 1 .LE. N .LE. 23,
C ZERO OTHERWISE.  FASTER BUT LESS MNEMONIC THAN MPPARA.
C FOR THE MEANINGS OF WORDS 1 TO 23 OF COMMON /MPCOM/, SEE
C THE COMMENTS IN SUBROUTINE MPPARM.  FOR THEIR DEFAULT
C SETTINGS SEE MPSET.
      COMMON /MPCOM/ MPC
      INTEGER MPC(23), N
      MPPARN = 0
      IF ((N .GT. 0) .AND. (N .LE. 23)) MPPARN = MPC(N)
      RETURN
      END
C $$                   ******  MPPI  ******
      SUBROUTINE MPPI (X)
C SETS MP X = PI TO THE AVAILABLE PRECISION.
C USES PI/4 = 4.ARCTAN(1/5) - ARCTAN(1/239), TIME = O(T**2).
C ROUNDING IS DEFINED BY THE PARAMETER RNDRL IN COMMON /MPCOM/ -
C RNDRL = 0 OR 1 - ERROR LESS THAN 0.6 UNITS IN LAST PLACE.
C RNDRL = 2 OR 3 - SEE COMMENTS IN SUBROUTINE MPNZR.
      COMMON R
      COMMON /MPCOM/ B, T, DUMMY
      INTEGER I2, I3, SV, TG
      INTEGER B, DUMMY(21), MPGD, R(1), T, X(1)
C SAVE T ETC.
      CALL MPSAVN (SV)
C SEE IF ANY GUARD DIGITS REQUIRED
      IF ((R(SV+2).NE.0).OR.(B.LE.100)) GO TO 10
C HERE NO GUARD DIGITS NECESSARY.
C ALLOCATE TEMPORARY SPACE.
      CALL MPNEW (I2)
      CALL MPART1 (5, R(I2))
      CALL MPMULI (R(I2), 4, R(I2))
      CALL MPART1 (239, X)
      CALL MPSUB (R(I2), X, X)
C USE ROUNDED MULTIPLICATION BY 4
      CALL MPSETR (1)
      CALL MPMULI (X, 4, X)
      GO TO 20
C HERE NEED TO USE SOME GUARD DIGITS
   10 T = T + MPGD (100)
      TG = T
      CALL MPNEW (I2)
      CALL MPNEW (I3)
      CALL MPART1 (5, R(I2))
      CALL MPMULI (R(I2), 4, R(I2))
C TAKE CARE IF DIRECTED ROUNDING REQUIRED.
      CALL MPREVR (1)
      CALL MPART1 (239, R(I3))
      CALL MPSETR (IABS(R(SV+2)))
      CALL MPSUB (R(I2), R(I3), R(I3))
      CALL MPMULI (R(I3), 4, R(I3))
C ROUND RESULT
      CALL MPMOVE (R(I3), TG, X, IABS(R(SV)))
C RESTORE EVERYTHING
   20 CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPPWR  ******
      SUBROUTINE MPPWR (X, N, Y)
C RETURNS Y = X**N, FOR MP X AND Y, INTEGER N, WITH 0**0 = 1.
C X MAY BE PACKED OR UNPACKED, Y IS UNPACKED.
C ROUNDING IS DEFINED BY THE PARAMETER RNDRL IN COMMON /MPCOM/ -
C RNDRL = 0 OR 1 - ERROR LESS THAN 0.6 UNITS IN LAST PLACE.
C RNDRL = 2 OR 3 - SEE COMMENTS IN SUBROUTINE MPNZR.
      COMMON R
      INTEGER I2, N, R(1), SV, TG, X(1), Y(1)
C SAVE T ETC. THEN INCREASE T AS NECESSARY.
      CALL MPSAVN (SV)
      CALL MPGD3 (N, TG)
C ALLOCATE TEMPORARY SPACE AND MOVE X.
      CALL MPNEW (I2)
      CALL MPMOVE (X, IABS(R(SV)), R(I2), TG)
      IF (N .GE. 0) GO TO 10
C N .LT. 0
      IF (X(1) .EQ. 0) CALL MPERRM (
     $  38HX = 0 AND N NEGATIVE IN CALL TO MPPWR$)
C HERE N .LT. 0,  X .NE. 0.  CHECK FOR DIRECTED ROUNDINGS.
      IF (MOD (IABS (N), 2) .EQ. 0) CALL MPREVR (-X(1))
      CALL MPREC (R(I2), R(I2))
      CALL MPSETR (IABS(R(SV+2)))
C NOW USE MPPWRA TO FIND R(I2)**ABS(N).
   10 CALL MPPWRA (R(I2), N, R(I2))
C ROUND RESULT
      CALL MPRND (R(I2), TG, Y, IABS(R(SV)), 0)
C RESTORE T ETC. AND RETURN.
      CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPPWRA  ******
      SUBROUTINE MPPWRA (X, N, Y)
C RETURNS Y = X**ABS(N) FOR MP X AND Y, INTEGER N.  (0**0 = 1)
C USES NO GUARD DIGITS, CHOPPED RATHER THAN ROUNDED ARITHMETIC.
C CALLED BY MPEXP, MPPWR, MPQPWR, MPROOT, ETC.,
C NOT RECOMMENDED FOR INDEPENDENT USE (USE MPPWR INSTEAD).
C ROUNDING NOT BEST POSSIBLE, BUT DIRECTED ROUNDING
C (RNDRL = 2 OR 3) IS CORRECT IN THAT LOWER AND UPPER BOUNDS
C (RESPECTIVELY) ARE OBTAINED.
      COMMON R
      INTEGER I2, N, N2, R(1), S, SV, X(1), Y(1)
C CHECK FOR SOME SPECIAL CASES.
      N2 = IABS(N)
      IF (N2 - 1) 10, 20, 30
C HERE ABS(N) = 0
   10 CALL MPCIM (1, Y)
      RETURN
C HERE ABS(N) = 1
   20 CALL MPSTR (X, Y)
      RETURN
C SAVE T ETC. AND ALLOCATE TEMPORARY SPACE ETC.
   30 CALL MPSAVN (SV)
      CALL MPNEW (I2)
C COMPUTE SIGN OF RESULT
      S = X(1)
      IF (MOD(N2,2).EQ.0) S = IABS(S)
C MOVE ABS(X) TO TEMPORARY STORAGE
      CALL MPABS (X, R(I2))
C INITIALIZE PRODUCT IN Y
      CALL MPCIM (1, Y)
C USE CHOPPED ARITHMETIC RATHER THAN ROUNDED TO SAVE TIME.
      IF (R(SV+2) .EQ. 1) CALL MPSETR (0)
C FOLLOWING FOR DIRECTED ROUNDING
      CALL MPREVR (-S)
C LOOP, LOOKING AT BITS OF ABS(N) FROM RIGHT.
   40 IF (MOD (N2, 2) .NE. 0) CALL MPMUL (Y, R(I2), Y)
      N2 = N2/2
      IF (N2.LE.0) GO TO 50
      CALL MPMUL (R(I2), R(I2), R(I2))
      GO TO 40
C RESTORE EVERYTHING AND FIX UP SIGN OF RESULT.
   50 CALL MPRESN (SV)
      Y(1) = S*Y(1)
      RETURN
      END
C $$                   ******  MPPWR2  ******
      SUBROUTINE MPPWR2 (X, Y, Z)
C RETURNS Z = X**Y FOR MP NUMBERS X, Y AND Z, WHERE X IS
C POSITIVE (X .EQ. 0 ALLOWED IF Y .GT. 0).  SLOWER THAN
C MPPWR AND MPQPWR, SO USE THEM IF POSSIBLE.
C X AND/OR Y MAY BE PACKED OR UNPACKED, Z IS UNPACKED.
C ROUNDING IS DEFINED BY THE PARAMETER RNDRL IN COMMON /MPCOM/ -
C RNDRL = 0 OR 1 - ERROR LESS THAN 0.6 UNITS IN LAST PLACE.
C RNDRL = 2 OR 3 - SEE COMMENTS IN SUBROUTINE MPNZR.
      COMMON R
      COMMON /MPCOM/ B, T, DUMMY
      INTEGER I2, I3, SV, TG
      INTEGER B, DUMMY(21), R(1), T, X(1), Y(1), Z(1)
      INTEGER MPCMPI, MPGD, MPGET, MPTLB
C SAVE T ETC., THEN CHECK SIGN OF X.
      CALL MPSAVN (SV)
      IF (X(1)) 10, 20, 30
   10 CALL MPERRM (29HX NEGATIVE IN CALL TO MPPWR2$)
C HERE X IS ZERO, RETURN ZERO IF Y POSITIVE, OTHERWISE ERROR
   20 IF (Y(1) .LE. 0) CALL MPERRM (
     $  43HX ZERO AND Y NONPOSITIVE IN CALL TO MPPWR2$)
C RETURN ZERO HERE
      Z(1) = 0
      GO TO 50
C HERE X IS POSITIVE.  CHECK FOR X = 1 OR Y = 0.
   30 IF ((Y(1) .NE. 0) .AND. (MPCMPI (X, 1) .NE. 0)) GO TO 40
C X**0 = 1, 1**Y = 1.
      CALL MPCIM (1, Z)
      GO TO 50
C USUAL CASE HERE, X POSITIVE, Y NONZERO.  ALLOCATE TEMPORARY
C SPACE AND USE MPLN AND MPEXP TO COMPUTE POWER.
C INCREASE T AS NECESSARY.
   40 T = T + 1 + MPGD (100*MPTLB(1))
     $      + MPGD (MPGET (X, 2)) + MAX0 (0, MPGET (Y, 2))
      TG = T
      CALL MPNEW (I2)
      CALL MPMOVE (X, IABS(R(SV)), R(I2), TG)
C TAKE CARE FOR DIRECTED ROUNDING
      CALL MPREVR (-Y(1))
      CALL MPLN (R(I2), R(I2))
      CALL MPNEW (I3)
      CALL MPMOVE (Y, IABS(R(SV)), R(I3), TG)
      CALL MPMUL (R(I2), R(I3), R(I2))
C RESTORE RNDRL BEFORE CALLING MPEXP
      CALL MPREVR (-Y(1))
C IF X**Y IS TOO LARGE, MPEXP WILL PRINT ERROR MESSAGE
      CALL MPEXP (R(I2), R(I2))
      CALL MPRND (R(I2), TG, Z, IABS(R(SV)), 0)
   50 CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPQPWR  ******
      SUBROUTINE MPQPWR (I, J, K, L, X)
C SETS MULTIPLE-PRECISION X = (I/J)**(K/L) FOR INTEGERS
C I, J, K AND L,  TIME = O(T**2).
C ROUNDING IS DETERMINED BY RNDRL IN COMMON /MPCOM/ -
C RNDRL = 0 OR 1 - ERROR LESS THAN 0.6 UNITS IN THE LAST PLACE,
C RNDRL = 2 OR 3 - DIRECTED ROUNDINGS, SEE SUBROUTINE MPNZR.
      COMMON R
      INTEGER IS, I2, JS, KS, LS, SV, TG
      INTEGER I, J, K, L, R(1), X(1)
C SAVE T ETC. AND INCREASE WORKING PRECISION
      CALL MPSAVN (SV)
      CALL MPGD3 (K, TG)
C ALLOCATE TEMPORARY STORAGE.
      CALL MPNEW (I2)
      R(I2) = 0
      IS = I
      JS = J
      KS = IABS(K)
      LS = L
C FOR EFFICIENCY MAKE KS POSITIVE AND LS NEGATIVE
C (SEE COMMENTS IN MPROOT AND MPPWR).
      IF (K) 30, 10, 40
C (I/J)**(0/L) = 1 IF J AND L ARE NONZERO.
   10 CALL MPCIM (1, R(I2))
      IF ((JS.NE.0).AND.(LS.NE.0)) GO TO 90
   20 CALL MPERRM (33HJ = 0 OR L = 0 IN CALL TO MPQPWR$)
C HERE K IS NEGATIVE
   30 LS = -LS
C NOW LOOK AT SIGN OF LS
   40 IF (LS) 60, 20, 50
C LS POSITIVE SO INTERCHANGE IS AND JS TO MAKE NEGATIVE
   50 IS = J
      JS = I
      LS = -LS
C NOW KS POSITIVE, LS NEGATIVE
   60 IF (IS .EQ. 0) CALL MPERRM (
     $  56HI=0 AND K/L.LT.0, OR J=0 AND K/L.GT.0 IN CALL TO MPQPWR$)
C (I/0)**(NEGATIVE) = 0 IF I NONZERO
      IF (JS.EQ.0) GO TO 90
C TO SAVE TIME IN MPROOT AND MPPWR, FIND GCD OF KS AND LS
      CALL MPGCD (KS, LS)
C CHECK FOR DIRECTED ROUNDING
      IF (MOD (KS, 2) .EQ. 0) CALL MPREVR (IS*(-(JS/IABS(JS))))
C CHECK FOR LS = -1, TREAT AS SPECIAL CASE
      IF (LS.NE.(-1)) GO TO 70
      CALL MPCQM (JS, IS, R(I2))
      GO TO 80
C USUAL CASE HERE, LS .NE. -1
   70 CALL MPCQM (IS, JS, R(I2))
      CALL MPROOT (R(I2), LS, R(I2))
   80 CALL MPSETR (IABS(R(SV+2)))
      CALL MPPWRA (R(I2), KS, R(I2))
C ROUND RESULT, RESTORE T ETC, AND RETURN
   90 CALL MPRND (R(I2), TG, X, IABS(R(SV)), 0)
      CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPREC  ******
      SUBROUTINE MPREC (X, Y)
C RETURNS Y = 1/X, FOR MP X AND Y.
C USES MPDIVL IF T SMALL OR RNDRL .NE. 0, MPROOT OTHERWISE.
C TIME = O(M(T)).
C ROUNDING DETERMINED BY RNDRL IN COMMON /MPCOM/ - SEE COMMENTS
C IN SUBROUTINE MPDIV.
      COMMON R
      INTEGER R(1), X(1), Y(1), CROSS, I2, SV
C FOLLOWING CROSSOVER POINT FOR USE OF MPROOT DETERMINED EMPIRICALLY.
      DATA CROSS /50/
      CALL MPSAVN (SV)
C CHECK FOR X ZERO
      IF (X(1) .EQ. 0) CALL MPERRM (
     $  44HATTEMPTED DIVISION BY ZERO IN CALL TO MPREC$)
C DECIDE WHETHER TO USE MPDIVL OR MPROOT.
      IF ((R(SV) .LT. CROSS) .OR. (R(SV+2) .NE. 0)) GO TO 10
      CALL MPROOT (X, -1, Y)
      GO TO 20
C HERE USE MPDIVL TO COMPUTE 1/X
   10 CALL MPNEW (I2)
      CALL MPCIM (1, R(I2))
      CALL MPDIVL (R(I2), X, Y)
   20 CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPRESN  ******
      SUBROUTINE MPRESN (N)
C RESTORES T, M AND RNDRL WHICH WERE SAVED BY
C SUBROUTINE MPSAVN.  N MUST BE THE VALUE RETURNED BY MPSAVN.
C SPTR IS RESTORED TO ITS VALUE ON THE CALL TO MPSAVN (I.E. N).
      COMMON /MPCOM/ B, T, M, LUN, MXR, SPTR, DUMMY
      INTEGER B, T, M, LUN, MXR, SPTR, N, DUMMY(17)
      CALL MPRES2 (N)
      SPTR = N
      RETURN
      END
C $$                   ******  MPRES2  ******
      SUBROUTINE MPRES2 (N)
C RESTORES T, M AND RNDRL WHICH WERE SAVED BY
C SUBROUTINE MPSAVN.  N MUST BE THE VALUE RETURNED BY MPSAVN.
C SPTR IS UNCHANGED.
      COMMON R
      COMMON /MPCOM/ B, T, M, LUN, MXR, SPTR, MXSPTR, MNSPTR,
     $  MXEXPN, MNEXPN, RNDRL, DUMMY
      INTEGER B, DUMMY(12), LUN, M, MNEXPN, MNSPTR, MXEXPN, MXR, MXSPTR,
     $  N, R(1), RNDRL, SPTR, T
      IF ((N .GE. MNSPTR) .AND. (SPTR .GE. (N+3))) GO TO 20
      WRITE (LUN, 10)
   10 FORMAT (44H *** ILLEGAL N OR STPR ON CALL TO MPRES2 ***)
      CALL MPERR
   20 T = R(N)
      M = R(N+1)
      RNDRL = R(N+2)
      RETURN
      END
C $$                   ******  MPREVR  ******
      SUBROUTINE MPREVR (I)
C IF I .GT. 0 AND RNDRL (IN COMMON /MPCOM/) IS 2 OR 3,
C RNDRL IS REPLACED BY 5 - RNDRL (I.E. 2 AND 3 INTERCHANGED).
C THIS REVERSES THE DIRECTION OF DIRECTED ROUNDING - SEE MPNZR.
      COMMON /MPCOM/ B, T, M, LUN, MXR, SPTR, MXSPTR, MNSPTR,
     $  MXEXPN, MNEXPN, RNDRL, DUMMY
      INTEGER B, DUMMY(12), I, LUN, M, MNEXPN, MNSPTR, MXEXPN, MXR,
     $  MXSPTR, RNDRL, SPTR, T
      IF ((I.GT.0) .AND. (RNDRL.GT.1)) RNDRL = 5 - RNDRL
      RETURN
      END
C $$                   ******  MPRND  ******
      SUBROUTINE MPRND (X, TX, Y, TY, K)
C MOVES X + S*K*(B**(-TY))*ABS(X) APPROPRIATELY ROUNDED TO Y,
C WHERE X IS AN MP NUMBER WITH TX DIGITS, Y IS AN MP NUMBER
C WITH TY DIGITS, K AN INTEGER, AND
C S = 0 IF RNDRL .LE. 1,
C    -1 IF RNDRL .EQ. 2,
C    +1 IF RNDRL .EQ. 3.
C NOTE - RNDRL = 0 HAS SAME EFFECT AS RNDRL = 1, I.E. X ROUNDED TO
C NEAREST REPRESENTABLE NUMBER WITH TY DIGITS.
      COMMON R
      COMMON /MPCOM/ B, T, M, LUN, MXR, SPTR, MXSPTR, MNSPTR,
     $  MXEXPN, MNEXPN, RNDRL, DUMMY
      INTEGER B, DUMMY(12), K, LUN, M, MNEXPN, MNSPTR, MXEXPN, MXR,
     $  MXSPTR, R(1), RNDRL, SPTR, T, TX, TY, X(1), Y(1), I2, SV
C SAVE T ETC., SET RNDRL TO 1 IF ZERO.
      CALL MPSAVN (SV)
      RNDRL = MAX0 (1, RNDRL)
C SEE IF ADDITION NECESSARY
      IF ((RNDRL.GT.1).AND.(K.NE.0).AND.(X(1).NE.0)) GO TO 10
      CALL MPMOVE (X, TX, Y, TY)
      GO TO 20
C HERE ADDITION NECESSARY.  ALLOCATE TEMPORARY SPACE ETC.
   10 T = TX
      CALL MPNEW (I2)
      CALL MPSTR (X, R(I2))
      R(I2+1) = R(I2+1) - TY
      CALL MPMULI (R(I2), ((2*RNDRL-5)*K*X(1)), R(I2))
      CALL MPADD (R(I2), X, R(I2))
      CALL MPMOVE (R(I2), TX, Y, TY)
C RESTORE EVERYTHING AND RETURN
   20 CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPROOT  ******
      SUBROUTINE MPROOT (X, N, Y)
C RETURNS Y = X**(1/N) FOR INTEGER N, MP X AND Y, USING NEWTONS
C METHOD WITHOUT DIVISIONS.
C X MAY BE PACKED OR UNPACKED, Y IS UNPACKED.
C TIME IS O(M(T)) UNLESS ABS(N) IS LARGE (WHEN MPPWR2 IS USED).
C ROUNDING DETERMINED BY PARAMETER RNDRL IN COMMON /MPCOM/ -
C RNDRL = 0 OR 1 - ERROR LESS THAN 0.6 UNITS IN LAST PLACE
C                  (SO RESULT EXACT IF EXACTLY REPRESENTABLE).
C RNDRL = 2 OR 3 - SEE COMMENTS IN SUBROUTINE MPNZR.
C ABS(N) .LE. M (MAXIMUM EXPONENT) IS REQUIRED.
      COMMON R
      COMMON /MPCOM/ B, T, M, DUMMY
      INTEGER E, EP, EX, I2, I3, I4, I5, I6, J, K, KS, NP, SV, TG, TG2
      INTEGER B, DUMMY(20), KTM, M, MPTLB, N, R(1), T, X(1), Y(1)
C SAVE T ETC., USE TRUNCATED ARITHMETIC INTERNALLY.
      CALL MPSAVN (SV)
      CALL MPSETR (0)
      NP = IABS (N)
      IF (N .NE. 1) GO TO 10
C SIMPLY MOVE X IF N = 1
      CALL MPUNPK (X, Y)
      GO TO 110
C CHECK FOR VARIOUS ILLEGAL ARGUMENT COMBINATIONS
   10 IF (N .EQ. 0) CALL MPERRM (24HN = 0 IN CALL TO MPROOT$)
      IF (X(1)) 30, 20, 40
   20 Y(1) = 0
      IF (N.LT.0) CALL MPERRM (37HX = 0 AND N .LT. 0 IN CALL TO MPROOT$)
      GO TO 110
   30 IF (MOD (NP, 2) .EQ. 0) CALL MPERRM (
     $  38HX .LT. 0 AND N EVEN IN CALL TO MPROOT$)
C INCREASE T AND ALLOCATE SPACE
   40 CALL MPGD3 (NP, TG)
      CALL MPNEW (I2)
      CALL MPNEW (I3)
C WORK WITH ABS(X), FIX UP SIGN LATER.
      CALL MPMOVE (X, IABS(R(SV)), R(I2), TG)
      R(I2) = IABS (R(I2))
C CHECK FOR LARGE ABS(N).
      IF (NP .GE. (M/4)) GO TO 130
      CALL MPNEW (I4)
      IF (NP .LE. 2) GO TO 50
      CALL MPNEW (I5)
      CALL MPNEW (I6)
   50 E = -R(I2+1)
C COMPUTE EP = FLOOR (E/NP) + 1
      EP = E/NP
      IF ((NP*EP) .LE. E) EP = EP + 1
C SCALE TO AVOID UNDER/OVERFLOW
C SCALED ABS(X) IS BETWEEN 1 AND B**NP
      R(I2+1) = R(I2+1) + EP*NP
C LOWER AND UPPER BOUNDS ON ABS(SCALED RESULT) ARE 1/B AND 1.
      CALL MPCQM (1, IABS(B), R(I3))
      CALL MPSTR (R(I3), R(I4))
      IF (NP .GT. 2) CALL MPCIM (1, R(I5))
C SET KTM TO MAXIMUM NUMBER OF ITERATIONS ALLOWED.
      KTM = T*MPTLB(1)
C REDUCE T AT FIRST.
      T = 2
      CALL MPGD3 (NP, TG2)
      EX = 0
C TO SPEED UP RECIPROCALS AND SQUARE ROOTS, TREAT NP = 1 OR 2 AS
C SPECIAL CASES AND GET BETTER STARTING APPROXIMATIONS FOR THEM.
      IF (NP .EQ. 1)
     $  CALL MPCQM (IABS(B), (B*R(I2+2) + R(I2+3) + 1), R(I3))
      IF (NP .NE. 2) GO TO 70
C HERE NP = 2.  COMPUTE J = UPPER BOUND ON 4*ABS(SCALED X).
      CALL MPCMI (R(I2), J)
      J = 4*(J + 1)
C COMPUTE K = UPPER BOUND ON SQRT(J)
C USING INTEGER NEWTON ITERATION.
      K = J
   60 KS = K
      K = (K + J/K)/2
      IF (K .LT. KS) GO TO 60
      IF ((K*K) .LT. J) K = K + 1
C NOW GET LOWER BOUND ON SCALED ABS (RESULT).
      CALL MPCQM (2, K, R(I3))
C SKIP BISECTION PHASE IF NP .LE. 2
   70 IF (NP .LE. 2) GO TO 80
C BISECTION LOOP HERE.  CHECK FOR INFINITE LOOP.
      KTM = KTM - 1
      IF (KTM .LT. 0) GO TO 120
      CALL MPSUB (R(I5), R(I3), R(I6))
      CALL MPDIVI (R(I6), 2, R(I6))
      CALL MPADD (R(I3), R(I6), R(I6))
      CALL MPPWRA (R(I6), NP, R(I4))
      CALL MPMUL (R(I2), R(I4), R(I4))
      CALL MPADDI (R(I4), -1, R(I4))
      IF (R(I4) .LE. 0) CALL MPSTR (R(I6), R(I3))
      IF (R(I4) .GE. 0) CALL MPSTR (R(I6), R(I5))
C REPEAT BISECTION IF ABS (RESIDUAL) .GE. 1/2
      IF (R(I4) .EQ. 0) GO TO 90
      IF ((R(I4+1) .GT. 0) .OR.
     $  ((R(I4+1) .EQ. 0) .AND. ((2*R(I4+2)) .GE. B))) GO TO 70
C NOW NEWTONS METHOD SHOULD CONVERGE
      CALL MPSTR (R(I6), R(I3))
      GO TO 90
C NEWTON LOOP STARTS HERE.  CHOOSE GOOD T.
   80 T = MIN0 (TG, TG2 + 4*IABS(EX))
      CALL MPPWRA (R(I3), NP, R(I4))
      CALL MPMUL (R(I2), R(I4), R(I4))
      CALL MPADDI (R(I4), -1, R(I4))
   90 EX = -T
      IF (R(I4) .NE. 0) EX = R(I4+1)
C CHECK FOR INFINITE LOOP
      KTM = KTM - 1
      IF ((KTM .LT. 0) .OR. (EX .GT. 0)) GO TO 120
      CALL MPMUL (R(I3), R(I4), R(I4))
      CALL MPDIVI (R(I4), NP, R(I4))
      CALL MPSUB (R(I3), R(I4), R(I3))
C CHECK FOR CONVERGENCE
      IF ((2*EX + TG) .GT. 0) GO TO 80
C END OF LOOP, CORRECT EXPONENTS
      R(I2+1) = R(I2+1) - EP*NP
      R(I3+1) = R(I3+1) + EP
C CORRECT RESULT FOR N POSITIVE
      IF (N .LT. 0) GO TO 100
      CALL MPPWRA (R(I3), N-1, R(I3))
      CALL MPMUL (R(I2), R(I3), R(I3))
  100 R(I3) = ISIGN (R(I3), X(1))
C RESTORE RNDRL AND DO CORRECT ROUNDING
      CALL MPSETR (IABS(R(SV+2)))
      CALL MPRND (R(I3), TG, Y, IABS(R(SV)), 1)
  110 CALL MPRESN (SV)
      RETURN
  120 CALL MPERRM (35HITERATION NOT CONVERGING IN MPROOT$)
C HERE ABS(N) IS VERY LARGE
  130 CALL MPCQM (1, N, R(I3))
      CALL MPPWR2 (R(I2), R(I3), R(I3))
      GO TO 100
      END
C $$                   ******  MPSAVN  ******
      SUBROUTINE MPSAVN (N)
C SAVES T, M AND RNDRL ON STACK.
C RETURNS N = OLD VALUE OF SPTR,
C SAVES T     IN R(N),
C       M     IN R(N+1),
C       RNDRL IN R(N+2).
C NOTE - THE ARGUMENT N MUST NOT HAVE SAME ADDRESS AS SPTR.
      COMMON R
      COMMON /MPCOM/ B, T, M, LUN, MXR, SPTR, MXSPTR, MNSPTR,
     $  MXEXPN, MNEXPN, RNDRL, DUMMY
      INTEGER B, DUMMY(12), LUN, M, MNEXPN, MNSPTR, MXEXPN, MXR, MXSPTR,
     $  N, R(1), RNDRL, SPTR, T
C ALLOCATE THREE WORDS ON STACK.
      CALL MPNEW2 (N, 3)
C SAVE T, M AND RNDRL.
      R(N) = T
      R(N+1) = M
      R(N+2) = RNDRL
      RETURN
      END
C $$                   ******  MPSCAL  ******
      SUBROUTINE MPSCAL (X, BASE, J)
C SETS  X = X*(BASE**J), FOR MP X, INTEGER J, AND (SMALL)
C POSITIVE INTEGER BASE (E.G. BASE = 10).
C ROUNDING NOT BEST POSSIBLE, BUT DIRECTED ROUNDING OPTIONS
C (RNDRL = 2 OR 3) GIVE CORRECT LOWER AND UPPER BOUNDS.
      COMMON R
      INTEGER B, I, IB, I2, JA, JP, SV
      INTEGER BASE, J, MPPARN, R(1), X(1)
      IF (BASE.LE.0) CALL MPERRM (31HILLEGAL BASE IN CALL TO MPSCAL$)
C RETURN IF J ZERO OR X ZERO
      IF ((J.EQ.0).OR.(X(1).EQ.0)) RETURN
C SAVE T ETC.
      CALL MPSAVN (SV)
C OTHERWISE MULTIPLY BY BASE**J
      JA = IABS(J)
C THE NUMBERS -500 AND 100 WERE DETERMINED EMPIRICALLY FOR BASE = 10.
C THE OPTIMUM CHOICE DEPENDS ON BASE, B AND T.
      IF ((J.GT.(-500)).AND.(J.LT.100)) GO TO 10
C HERE EXPONENT LARGE, SO USE MPPWRA TO COMPUTE BASE**ABS(J)
      CALL MPNEW (I2)
      CALL MPCIM (BASE, R(I2))
C TAKE CARE FOR DIRECTED ROUNDINGS.
      CALL MPREVR ((-J)*X(1))
      CALL MPPWRA (R(I2), JA, R(I2))
      CALL MPSETR (IABS(R(SV+2)))
      IF (J.LT.0) CALL MPDIV (X, R(I2), X)
      IF (J.GE.0) CALL MPMUL (X, R(I2), X)
      GO TO 30
C HERE ABS(J) IS SMALL SO PROBABLY FASTER TO USE MPDIVI OR MPMULI
   10 JP = 1
      IB = MPPARN (16)/BASE
      B = MPPARN (1)
      DO 20 I = 1, JA
      JP = BASE*JP
      IF ((JP.LE.IB).AND.(JP.NE.B).AND.(I.LT.JA)) GO TO 20
      IF (J.LT.0) CALL MPDIVI (X, JP, X)
      IF (J.GE.0) CALL MPMULI (X, JP, X)
      JP = 1
   20 CONTINUE
   30 CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPSETR  ******
      SUBROUTINE MPSETR (N)
C SETS THE PARAMETER RNDRL IN COMMON /MPCOM/ TO N = 0, 1, 2 OR 3.
C FOR EFFECT OF THIS SEE SUBROUTINE MPNZR ETC.
      INTEGER N
      IF ((N .LT. 0) .OR. (N .GT. 3)) CALL MPERRM (
     $  35HILLEGAL ARGUMENT IN CALL TO MPSETR$)
      CALL MPPARC (11, N)
      RETURN
      END
C $$                   ******  MPSET2  ******
      SUBROUTINE MPSET2 (LUN, DECPL, MT2, MNSPTR, MXR)
C
C LUN IS THE LOGICAL UNIT TO BE USED FOR ERROR MESSAGES.
C SETS BASE (B) AND NUMBER OF DIGITS (T) TO GIVE THE
C EQUIVALENT OF AT LEAST DECPL DECIMAL DIGITS.
C DECPL SHOULD BE A POSITIVE INTEGER.
C MT2 IS THE DIMENSION OF ARRAYS USED FOR MP NUMBERS,
C SO AN ERROR OCCURS IF THE COMPUTED T EXCEEDS MT2 - 2.
C MXR IS THE SIZE OF BLANK COMMON DECLARED IN THE CALLING PROGRAM,
C AND MNSPTR IS THE FIRST LOCATION IN BLANK COMMON AVAILABLE FOR
C USE BY MP.  (MP WILL USE WORDS MNSPTR, MNSPTR+1, ... , MXR OF
C BLANK COMMON (COUNTING FROM 1) FOR WORKING STORAGE.)
C
C LET MXINT BE THE LARGEST INTEGER OF THE FORM 2**K - 1 REPRESENTABLE
C AS A SIGNED INTEGER ON THE MACHINE (SEE MPLARG).
C THE COMPUTED B AND T SATISFY THE CONDITIONS
C B**(T-1) .GE. 10**(DECPL-1)  AND 8*B*B-1 .LE. MXINT .
C APPROXIMATELY MINIMAL T AND MAXIMAL B SATISFYING
C THESE CONDITIONS ARE CHOSEN.
C
C MPSET2 ALSO SETS
C
C       M      = MXINT/4 (MAXIMUM ALLOWABLE EXPONENT),
C       SPTR   = MNSPTR  (STACK POINTER),
C       MXSPTR = MNSPTR  (MAXIMUM VALUE OF SPTR IS SAVED HERE),
C       RNDRL  = 0       (SEE SUBROUTINE MPNZR),
C       KTUNFL = 0       (UNDERFLOW COUNTER),
C       MXUNFL = 0       (SEE SUBROUTINE MPUNFL),
C       MNEXPN = M+1     (MINIMUM EXPONENT WHICH OCCURS SAVED HERE),
C       MXEXPN = -M      (MAXIMUM EXPONENT WHICH OCCURS SAVED HERE),
C       MXINT  = LARGE INTEGER (SEE SUBROUTINE MPLARG),
C       EXWID  = 6       (EXPONENT FIELD WIDTH, SEE MPFOUT),
C       INRECL = 80      (INPUT RECORD LENGTH, SEE MPFIN),
C       INBASE = 10      (DEFAULT INPUT BASE FOR MPIN),
C       OUTBAS = 10      (DEFAULT OUTPUT BASE FOR MPOUT),
C       EXPCH  = 1HE     (DEFAULT OUTPUT EXPONENT CHARACTER),
C       CHWORD = NUMBER OF CHARACTERS PER WORD (RETURNED BY MPUPW),
C       ONESCP = 1 IF ONES COMPLEMENT ARITHMETIC, 0 OTHERWISE.
C
C IF THESE ARE NOT ALL AS DESIRED, CHANGE AFTER THE CALL TO MPSET2,
C FOR EXAMPLE, ONE COULD SET MXUNFL = 10 IF
C EXECUTION WERE TO BE TERMINATED AFTER 10 MP UNDERFLOWS.
C
      COMMON R
      COMMON /MPCOM/ B, T, M, LUNC, MXRC, SPTR, MXSPTR, MNSPTC,
     $  MXEXPN, MNEXPN, RNDRL, KTUNFL, MXUNFL,
     $  DECPLC, MT2C, MXINT, EXWID, INRECL, INBASE,
     $  OUTBAS, EXPCH, CHWORD, ONESCP
C INCREASE DIMENSION OF C AND MODIFY DATA STATEMENT FOR MAXCHW
C BELOW IF THERE ARE MORE THAN 10 CHARACTERS PER WORD.
C 10 CHARACTERS PER WORD.
      INTEGER C(10), MAXCHW,
     $  I, I2, K, TAB(10),
     $  B, CHW, CHWORD, DECPL, DECPLC, EXPCH, EXWID, INBASE, INRECL,
     $  KTUNFL, LUN, LUNC, M, MNEXPN, MNSPTC, MNSPTR, MT2, MT2C,
     $  MXEXPN, MXINT, MXR, MXRC, MXSPTR, MXUNFL, ONESCP, OUTBAS,
     $  R(1), RNDRL, SPTR, T, MPDIGS, MPGET
      LOGICAL MPIS
C SEE COMMENT ABOVE ABOUT MAXCHW.
      DATA MAXCHW /10/
      DATA TAB(1), TAB(2), TAB(3), TAB(4) /1H0, 1H9, 1HA, 1HZ/
      DATA TAB(5), TAB(6), TAB(7), TAB(8) /1H+, 1H-, 1H., 1H$/
      DATA TAB(9), TAB(10) /1H , 1HE/
C FIRST SET OUTPUT UNIT, MAXIMUM SIZE OF COMMON, ETC.
      LUNC = LUN
      MXRC = MXR
      MNSPTC = MNSPTR
      SPTR = MNSPTR
      MXSPTR = SPTR
      RNDRL = 0
      KTUNFL = 0
      MXUNFL = 0
      DECPLC = DECPL
      MT2C = MT2
C SET DEFAULT IO PARAMETERS
      EXWID = 6
      INRECL = 80
      INBASE = 10
      OUTBAS = 10
      EXPCH = MPGET (TAB, 10)
C CHECK THAT LUN IN RANGE 1,...,99 AND WRITE ERROR MESSAGE
C ON UNIT 6 IF NOT.
      IF ((LUN.GT.0).AND.(LUN.LT.100)) GO TO 20
      WRITE (6, 10) LUN
   10 FORMAT (10H *** LUN =, I10, 30H ILLEGAL IN CALL TO MPSET2 ***)
      LUNC = 6
      CALL MPERR
C CHECK FOR BURROUGHS 6700
   20 IF (.NOT. MPIS (TAB(9), TAB(6))) GO TO 40
   25 WRITE (LUNC, 30)
   30 FORMAT (35H *** REPLACE MPIS AND/OR MPLARG ***)
      CALL MPERR
C SET MXINT TO LARGE REPRESENTABLE INTEGER (SEE ABOVE).
   40 CALL MPLARG (MXINT)
C IF WORDLENGTH .GE. 12 BITS, EXPECT MXINT .GE. 2047
      IF (MXINT .LT. 2047) GO TO 25
C MAXIMUM EXPONENT IS MXINT/4
      M = MXINT/4
C BASE B IS LARGEST POWER OF 2 SUCH THAT 8*B*B-1 .LE. MXINT
      B = 1
   50 B = 2*B
      IF ((4*B*B) .LT. (M+1)) GO TO 50
C SET MNEXPN AND MXEXPN TO GREATER THAN M, LESS THAN -M RESP.
      MNEXPN = M+1
      MXEXPN = -M
      IF (DECPL.GT.0) GO TO 70
      WRITE (LUNC, 60)
   60 FORMAT (39H *** DECPL .LE. 0 IN CALL TO MPSET2 ***)
      CALL MPERR
C ENSURE THAT B**(T-1) .GE. 10**(DECPL-1).
   70 T = MPDIGS (DECPL)
C SEE IF T TOO LARGE FOR DIMENSION STATEMENTS
      I2 = T + 2
      IF (I2.LE.MT2) GO TO 90
      WRITE (LUNC, 80) I2
   80 FORMAT (40H *** MT2 TOO SMALL IN CALL TO MPSET2 *** /
     $  45H *** INCREASE MT2 AND DIMENSIONS OF MP ARRAYS,
     $  12H TO AT LEAST, I6, 4H ***)
      CALL MPERR
      CHWORD = MAXCHW
C NOW CHOOSE ONESCP SO THAT MPUPW WORKS FOR ALL
C CHARACTERS OF INTEREST (SEE MPIN, MPOUT ETC).
   90 ONESCP = 0
      DO 120 K = 1, 2
      DO 100 I = 1, 9
      CALL MPUPW (TAB(I), C, CHW)
      CHWORD = CHW
      IF ((CHWORD .LE. 1) .OR. (CHWORD .GT. MAXCHW)) GO TO 120
      IF (.NOT. MPIS (TAB(I), C(1))) GO TO 110
      IF (.NOT. MPIS (TAB(9), C(CHWORD))) GO TO 110
  100 CONTINUE
      GO TO 120
  110 ONESCP = ONESCP + 1
  120 CONTINUE
      IF (ONESCP .LE. 1) GO TO 140
      WRITE (LUNC, 130)
  130 FORMAT (22H *** REPLACE MPUPW ***)
      CALL MPERR
C CHECK LEGALITY OF B, T ETC.
  140 CALL MPCHK
      RETURN
      END
C $$                   ******  MPSTOV  ******
      SUBROUTINE MPSTOV (N)
C
C CALLED IF STACK SPACE IN BLANK COMMON TOO SMALL.
C IF POSSIBLE, SPACE SHOULD BE EXPANDED BY AT LEAST N WORDS
C AND MXR INCREASED BY THE AMOUNT EXPANDED.
C
C THE NEW SPACE MUST BE CONTIGUOUS WITH R(MXR), I.E. IT MUST
C INCLUDE LOCATIONS R(MXR+1), ... , R(MXR+N).
C
C AT PRESENT MPSTOV DOES NOTHING BECAUSE THE METHOD OF EXPANDING
C THE STACK IS MACHINE AND OPERATING-SYSTEM DEPENDENT.
C
      COMMON R
      COMMON /MPCOM/ B, T, M, LUN, MXR, DUMMY
      INTEGER B, DUMMY(18), LUN, M, MXR, N, R(1), T
      WRITE (LUN, 10)
   10 FORMAT (42H *** MPSTOV CALLED BUT NOT IMPLEMENTED ***)
      RETURN
      END
C $$                   ******  MPSTR  ******
      SUBROUTINE MPSTR (X, Y)
C SETS Y = X FOR MP X AND Y.
C X MAY BE IN PACKED OR UNPACKED FORMAT,
C Y WILL BE IN SAME FORMAT AS X.
      COMMON /MPCOM/ B, T, DUMMY
      INTEGER B, T, DUMMY(21), I, J, X(1), Y(1), T2
C SEE IF X AND Y HAVE THE SAME ADDRESS (THEY OFTEN DO)
C NOTE - THIS TEST IS FOR EFFICIENCY ONLY, AND IT DOES NOT MATTER
C IF THE FACT THAT X AND Y HAVE THE SAME ADDRESS IS NOT DETECTED.
      J = X(1)
      Y(1) = J + 1
      IF (J.EQ.X(1)) GO TO 10
C HERE X(1) AND Y(1) MUST HAVE THE SAME ADDRESS
      X(1) = J
      RETURN
C HERE X(1) AND Y(1) MAY HAVE DIFFERENT ADDRESSES
   10 Y(1) = J
C NO NEED TO MOVE X(2), ... IF X(1) = 0
      IF (J.EQ.0) RETURN
      T2 = T + 2
C REDUCE T2 IF X IS IN PACKED FORMAT.
      IF (IABS(J) .GT. 1) T2 = (T+3)/2
C NOW MOVE REMAINING WORDS OF X.
      DO 20 I = 2, T2
   20 Y(I) = X(I)
      RETURN
      END
C $$                   ******  MPSUB  ******
      SUBROUTINE MPSUB (X, Y, Z)
C SUBTRACTS Y FROM X, FORMING RESULT IN Z, FOR MP X, Y AND Z.
C ROUNDING IS CONTROLLED BY THE PARAMETER RNDRL IN COMMON /MPCOM/
C AS FOLLOWS -
C
C RNDRL = 0 - ROUND TOWARDS ZERO IF X*Y .LE. 0,
C             AWAY FROM ZERO IF X*Y .GT. 0,
C             IN BOTH CASES ONE GUARD DIGIT IS USED, SO RESULT
C             IS EXACT IF SIGNIFICANT CANCELLATION OCCURS.
C
C RNDRL = 1, 2 OR 3 - SEE COMMENTS IN SUBROUTINE MPNZR.
C
      INTEGER X(1), Y(1), Z(1), S
C SAVE SIGN OF X
      S = X(1)
C REVERSE SIGN OF Y, SEE IF SIGN OF X CHANGED
      Y(1) = -Y(1)
      IF (S.EQ.X(1)) GO TO 10
C HERE X AND Y HAVE THE SAME ADDRESS SO RESULT IS ZERO
      Y(1) = S
      Z(1) = 0
      RETURN
C HERE X AND Y HAVE DIFFERENT ADDRESSES (OR BOTH ZERO) SO USE MPADD.
   10 CALL MPADD (X, Y, Z)
C RESTORE SIGN OF Y, BEING CAREFUL IN CASE Y AND Z HAVE SAME ADDRESS
      S = Z(1)
      Y(1) = -Y(1)
      Z(1) = S
      RETURN
      END
C $$                   ******  MPSUBA  ******
      SUBROUTINE MPSUBA (X, Y, I)
C CALLED BY MPLARG AND MPUPW.
      INTEGER I, X(1), Y(1)
      I = MAX0 (1, X(1) - Y(1))
      RETURN
      END
C $$                   ******  MPTLB  ******
      INTEGER FUNCTION MPTLB (J)
C RETURNS UPPER BOUND ON ABS(J)*LOG2(B).
      INTEGER J, MPCHGB, MPPARN
      MPTLB = MPCHGB (2, MPPARN(1), IABS(J))
      RETURN
      END
C $$                   ******  MPUNFL  ******
      SUBROUTINE MPUNFL (X)
C CALLED ON MULTIPLE-PRECISION UNDERFLOW, IE WHEN THE
C EXPONENT OF MP NUMBER X WOULD BE LESS THAN 1-M.
C ACTION DEPENDS ON PARAMETERS RNDRL AND MXUNFL IN COMMON /MPCOM/ -
C IF RNDRL .LE. 1 THEN
C   IF MXUNFL .EQ. 0 (DEFAULT OPTION IF MPSET CALLED), X IS SET
C                     TO ZERO AND EXECUTION CONTINUES,
C   IF MXUNFL .GT. 0, X IS SET TO ZERO AND EXECUTION CONTINUES
C                     UNLESS MXUNFL UNDERFLOWS HAVE OCCURRED,
C                     WHEN EXECUTION IS TERMINATED BY A CALL
C                     TO MPERR.
C IF RNDRL .EQ. 2, ACTION AS ABOVE EXCEPT X IS SET TO
C   -(SMALLEST POSITIVE REPRESENTABLE NUMBER).
C IF RNDRL .EQ. 3, ACTION AS ABOVE EXCEPT X IS SET TO
C   +(SMALLEST POSITIVE REPRESENTABLE NUMBER).
      COMMON /MPCOM/ B, T, M, LUN, MXR, SPTR, MXSPTR, MNSPTR,
     $  MXEXPN, MNEXPN, RNDRL, KTUNFL, MXUNFL, DUMMY
      INTEGER B, DUMMY(10), KTUNFL, LUN, M, MNEXPN, MNSPTR, MXEXPN, MXR,
     $  MXSPTR, MXUNFL, RNDRL, SPTR, T, X(1)
C SINCE M MAY HAVE BEEN OVERWRITTEN, CHECK B, T, M ETC.
      CALL MPCHK
C THE UNDERFLOWING NUMBER IS SET TO ZERO
      X(1) = 0
C SET X TO -+ SMALLEST POSITIVE REPRESENTABLE NUMBER IF
C RNDRL = 2 OR 3 RESPECTIVELY.
      IF (RNDRL.GE.2) CALL MPMINR (X)
      IF (RNDRL.EQ.2) X(1) = -X(1)
C MINIMUM EXPONENT INDICATOR (MNEXPN) IS SET TO -M
      MNEXPN = -M
C INCREMENT UNDERFLOW COUNTER (KTUNFL) AND CALL MPERR IF
C MXUNFL UNDERFLOWS HAVE OCCURRED.
      KTUNFL = KTUNFL + 1
      IF (KTUNFL.NE.MXUNFL) RETURN
      WRITE (LUN, 10) KTUNFL
   10 FORMAT (4H ***, I10, 32H MP UNDERFLOWS HAVE OCCURRED ***)
      CALL MPERR
      RETURN
      END
C $$                   ******  MPUNPK  ******
      SUBROUTINE MPUNPK (Y, X)
C RESTORES THE MP NUMBER X WHICH IS STORED IN PACKED OR
C UNPACKED FORMAT IN THE INTEGER ARRAY Y.  FOR FURTHER
C DETAILS SEE SUBROUTINE MPPACK.
      COMMON /MPCOM/ B, T, DUMMY
      INTEGER I, IB, J, TWO
      INTEGER B, DUMMY(21), T, X(1), Y(1)
C CHECK IF Y IS PACKED OR NOT.
      IF (IABS(Y(1)) .GT. 1) GO TO 10
C HERE Y IS ZERO OR UNPACKED.
      CALL MPSTR (Y, X)
      RETURN
C HERE Y IS NONZERO AND PACKED.
   10 TWO = 2
      J = T/2
C DO LAST DIGIT SEPARATELY IF T ODD.
      IF (MOD (T, 2) .NE. 0) X(T+2) = Y(J+2)/B
C WORK BACKWARDS IN CASE X AND Y ARE THE SAME ARRAY.
      IF (J .LT. 2) GO TO 30
      DO 20 IB = 2, J
      I = J - IB
      X(2*I+6) = MOD (Y(I+3), B)
   20 X(2*I+5) = Y(I+3)/B
C FIRST WORD OF Y MAY BE SIGNED.
   30 X(TWO+2) = MOD (IABS(Y(1)), B)
      X(TWO+1) = IABS(Y(1))/B
C MOVE SIGN AND EXPONENT
      X(TWO) = Y(TWO)
      X(1) = ISIGN (1, Y(1))
      RETURN
      END
C $$                   ******  MPUPDT  ******
      SUBROUTINE MPUPDT (J)
C UPDATES MINIMUM AND MAXIMUM EXPONENT INDICATORS,
C ASSUMING J IS THE EXPONENT OF A NEWLY FORMED NONZERO MP NUMBER.
C NOTE - MPUPDT DOES NOT CHECK FOR MP OVERFLOW OR UNDERFLOW.
      COMMON /MPCOM/ B, T, M, LUN, MXR, SPTR, MXSPTR, MNSPTR,
     $  MXEXPN, MNEXPN, DUMMY
      INTEGER B, DUMMY(13), J, LUN, M, MNEXPN, MNSPTR, MXEXPN, MXR,
     $  MXSPTR, SPTR, T
      MNEXPN = MIN0 (J, MNEXPN)
      MXEXPN = MAX0 (J, MXEXPN)
      RETURN
      END
C $$                   ******  MPUPK  ******
      SUBROUTINE MPUPK (SOURCE, DEST, LDEST, LFIELD)
C
C THIS SUBROUTINE UNPACKS A PACKED HOLLERITH STRING (SOURCE)
C PLACING ONE CHARACTER PER WORD IN THE ARRAY DEST (AS IF READ IN
C A1 FORMAT). IT CONTINUES UNPACKING UNTIL IT FINDS A SENTINEL ($)
C OR UNTIL IT HAS FILLED LDEST WORDS OF THE ARRAY DEST.
C THE LENGTH OF THE UNPACKED STRING (EXCLUDING THE SENTINEL IF
C ANY) IS RETURNED IN LFIELD.  THUS 0 .LE. LFIELD .LE. LDEST. GT. 0
C
      COMMON R
      INTEGER DEST(1), I, I2, J, K, K2, LDEST, LFIELD, MPPARN, N,
     $  R(1), SENTNL, SOURCE(1), SV, DOLLAR
      LOGICAL MPIS
      DATA DOLLAR /1H$/
C REPLACE $ BY COMPILER-GENERATED END-OF-STRING SENTINEL (IF ANY).
      DATA SENTNL /1H$/
      CALL MPSAVN (SV)
C MPPARN(22) SHOULD RETURN UPPER BOUND ON THE NUMBER OF CHARACTERS
C PER WORD (SEE MPSET2).
      CALL MPNEW2 (I2, MPPARN(22))
      I = 0
      J = 0
C LOOP TO UNPACK EACH WORD OF SOURCE
   10 I = I + 1
      CALL MPUPW (SOURCE (I), R(I2), N)
C ASSUME END OF STRING IF N .LE. 0
      IF (N .LE. 0) GO TO 30
C MOVE EACH CHARACTER TO DEST
      DO 20 K = 1, N
      K2 = I2 + K - 1
C CHECK FOR SENTINEL OR MAXIMUM LENGTH OF DEST
      IF (MPIS (R(K2), DOLLAR) .OR. MPIS (R(K2), SENTNL) .OR.
     $  (J .GE. LDEST)) GO TO 30
      J = J + 1
   20 DEST(J) = R(K2)
      IF (J .LT. LDEST) GO TO 10
   30 LFIELD = J
      CALL MPRESN (SV)
      RETURN
      END
C $$                   ******  MPUPW  ******
      SUBROUTINE MPUPW (W, C, N)
C
C              *************************    (SEE THE MP
C              *** MACHINE DEPENDENT ***     USERS GUIDE FOR
C              *************************     CONVERSION HINTS)
C
C WHEN CALLED WITH PACKED CHARACTER STRING IN W,
C RETURNS N = NUMBER OF CHARACTERS/WORD AND THE
C CHARACTERS IN C(1), ... , C(N), UNPACKED, LEFT JUSTIFIED,
C BLANK FILLED.
C
C DOES NOT WORK ON MACHINES SUCH AS CYBER 76 AND BURROUGHS 6700
C WHICH DO NOT PERFORM INTEGER ARITHMETIC ON FULL WORDS, OR
C ON MACHINES SUCH AS PDP 11 WHICH STORE CHARACTERS IN UNUSUAL
C ORDER, OR ON MACHINES WITH DECIMAL OR SIGN-AND-MAGNITUDE
C ARITHMETIC. DOES WORK ON UNIVAC 1100, DEC 10 AND MANY OTHER
C MACHINES, THOUGH.
C
      INTEGER BLANK, BMK, BMK2, K, KL, KLM, KM, K2, L, M, ONESCP,
     $  BL2, C(1), MPPARN, N, W, WS, WSM
      DATA BL2 /1H /
      BLANK = BL2
C WORD 23 OF COMMON /MPCOM/ IS SET TO 0 OR 1 BY MPSET2.
      ONESCP = MPPARN (23)
C ASSUME CHARACTERS 0 AND 1 HAVE CONSECUTIVE BINARY CODES
C AND AT LEAST 2 CHARACTERS PER WORD.
      CALL MPSUBA (2H10, 2H00, K)
      KM = K - ONESCP
      K2 = K/2
C COMPUTE L = 2**(BITS PER CHARACTER)
      CALL MPSUBA (2H01, 2H00, L)
      L = MAX0 (1, K/L)
      KL = K*(L/4)
      KLM = KL - ONESCP
      M = K
      N = 0
      WS = W
C STRIP OFF LEFTMOST BLANK
      IF (BLANK .GE. 0) BMK = MOD (BLANK, K)
      IF (BLANK .LT. 0) BMK = MOD (BLANK + KLM, K)
      BMK2 = BMK - KL
C LOOP FOR EACH CHARACTER IN WORD
   10 N = N + 1
C CHECK SIGN OF WS TO AVOID INTEGER OVERFLOW
      IF (WS .LT. 0) GO TO 20
C GET LEFTMOST CHARACTER OF WS, PAD WITH BLANKS
      C(N) = K*(WS/K) + BMK
      GO TO 30
C HERE WS NEGATIVE
C NOTE THAT (KL + KLM) MIGHT OVERFLOW, SO ADD IN TWO STEPS
   20 WS = (WS + KL) + KLM
      C(N) = (K*(WS/K) + BMK2) - KLM
C STRIP OFF LEFTMOST CHARACTER OF WS
   30 WSM = MOD (WS, K)
C AND SHIFT LEFT ONE CHARACTER, AVOIDING OVERFLOW
      IF (WSM .LT. K2) WS = L*WSM
      IF (WSM .GE. K2) WS = L*(WSM - K) + ONESCP
      M = M/L
C LOOP AT MOST 10 TIMES (CHANGE IF MACHINE HAS MORE
C THAN TEN CHARACTERS PER WORD).
      IF ((M .GT. 0) .AND. (N .LT. 10)) GO TO 10
      RETURN
      END
